# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

**OG Shortlink** - PHP 8.3 URL shortener with OpenGraph metadata support, multi-domain management, IP intelligence, and Facebook bot protection. Deployed on cPanel shared hosting.

## Tech Stack

- **PHP 8.3** with strict_types, PSR-4 autoloading
- **MariaDB 10.11+** via PDO (prepared statements only)
- **APCu** for caching (Redis optional)
- **Frontend**: Vanilla HTML/CSS/JS, jQuery

## Development Commands

```bash
composer install              # Install dependencies
composer stan                 # PHPStan static analysis (level max)
composer cs                   # PHPCS PSR-12 check
composer cs:fix               # Auto-fix code style
composer fixer:dry            # Preview PHP CS Fixer changes
composer quality              # Run stan + cs together
composer test                 # PHPUnit tests
```

## Architecture

This project follows **Clean Architecture** with layers:

```
src/
├── Domain/                   # Business logic, entities, interfaces
│   └── Domain/               # Domain repository interface
│
├── Application/              # Use cases, orchestration
│   └── Shortlink/            # ShortlinkResolver service
│
├── Infrastructure/           # External concerns, implementations
│   ├── Cache/                # CacheInterface, ApcuCache, FileCache
│   ├── Http/                 # Request, Response, JsonResponse, RedirectResponse
│   └── Persistence/          # RepositoryInterface, AbstractDomainRepository
│
└── Presentation/             # HTTP handlers
    ├── Controller/           # RedirectController, OgPageRenderer
    │   └── Api/              # API controllers (modular)
    │       ├── ApiRouter.php
    │       ├── BaseApiController.php
    │       ├── ShortlinkApiController.php
    │       ├── UserApiController.php
    │       ├── DomainApiController.php
    │       ├── SettingsApiController.php
    │       ├── AnalyticsApiController.php
    │       ├── ProtectionApiController.php
    │       ├── EnvApiController.php
    │       └── ThirdPartyApiController.php
    └── Middleware/           # MiddlewareInterface

app/                          # Legacy code (stable, not actively migrated)
├── Admin/                    # Admin auth (Auth.php, Security.php)
├── Analytics/                # Device/GeoIP detection, traffic logging
├── Database/                 # Connection pooling
├── IpIntel/                  # IP risk scoring (multiple providers)
├── Monitoring/               # Health checks
├── GlobalDomainRepository.php  # Extends AbstractDomainRepository
├── UserDomainRepository.php    # Extends AbstractDomainRepository
└── ShortlinkRepository.php

public/                       # Web root
├── index.php                 # Main redirect handler (uses RedirectController)
├── health.php                # Health check endpoint
├── shim.php                  # Bot protection (v1)
├── shim_v2.php               # Bot protection (v2)
├── api/shortlinks.php        # REST API (external)
└── admin/
    ├── api.php               # Admin API (uses ApiRouter)
    └── [dashboard pages]
```

## Autoloading

Both `src/` and `app/` use `App\` namespace (PSR-4):
```json
"autoload": {
    "psr-4": {
        "App\\": ["src/", "app/"]
    }
}
```

## Key Entry Points

### 1. Redirect Handler (`public/index.php`)
```php
$request = Request::fromGlobals();
$controller = new RedirectController();
$response = $controller->handle($request);
$response->send();
```

### 2. Admin API (`public/admin/api.php`)
```php
$router = new ApiRouter();
$response = $router->route($action);
$response->send();
```

## Database Tables

- `shortlinks` - Core shortlink data with OG metadata, domain reference
- `traffic_analytics` - Per-request analytics (IP, geo, device, risk)
- `traffic_summary_daily` - Aggregated daily stats
- `global_domains` - Admin-managed domains with DNS integration
- `user_domains` - Per-user domain configurations
- `users` - Admin/user accounts
- `settings` - Key-value application config

## New Abstractions

### Cache Interface
```php
use App\Infrastructure\Cache\CacheInterface;
use App\Infrastructure\Cache\ApcuCache;

$cache = new ApcuCache('prefix:');
$cache->get('key');
$cache->set('key', $value, 300);
```

### HTTP Abstractions
```php
use App\Infrastructure\Http\Request;
use App\Infrastructure\Http\JsonResponse;
use App\Infrastructure\Http\RedirectResponse;

$request = Request::fromGlobals();
$response = JsonResponse::success(['data' => $data]);
$redirect = RedirectResponse::temporary($url);
```

### Repository Pattern
```php
use App\Infrastructure\Persistence\AbstractDomainRepository;

// Repositories extend AbstractDomainRepository for shared logic:
// - buildUpdateQuery()
// - executeInsert(), executeUpdate(), executeDelete()
// - fetchOne(), fetchAll(), rowExists()
```

### API Controllers
```php
use App\Presentation\Controller\Api\ShortlinkApiController;

// Each controller extends BaseApiController with:
// - readJson() - Parse JSON body
// - checkRateLimit() - Rate limiting
// - requireAdmin() - Admin check
// - verifyCsrf() - CSRF validation
// - success(), error() - Response helpers
```

## Security Requirements

- `declare(strict_types=1);` in all PHP files
- Prepared statements for ALL database queries
- `htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8')` for ALL output
- CSRF tokens for all POST forms (rotated after login)
- Security headers (CSP, HSTS, X-Frame-Options)
- API key validation (503 if SHORTLINK_API_KEY not configured)

## Configuration

Environment variables in `.env`:
- `DB_*` - Database connection
- `APP_ENV`, `APP_DEBUG`, `APP_URL`
- `ADMIN_USERNAME`, `ADMIN_PASSWORD_HASH` (bcrypt)
- `SHORTLINK_API_KEY` - API authentication (required, 503 if missing)
- `IP_INTEL_ENABLED`, `RISK_THRESHOLD` - IP risk scoring
- `SHIM_HMAC_SECRET` - Bot protection URL signing

## API Endpoints

- `POST /api/shortlinks.php` - Create shortlink (requires `X-Shortlink-Key` header)
- `POST /admin/api.php?action=...` - Admin operations (session auth + CSRF)
- `GET /health.php` - System health check

## Refactoring Status

**Completed:**
- Phase 1: Security fixes (CSRF rotation, SQL injection, debug logging, API key validation)
- Phase 2: Quality gate setup (phpstan.neon, phpcs.xml)
- Phase 3: New src/ directory structure with Clean Architecture
- Phase 4: Core abstractions (Cache, HTTP, Middleware interfaces)
- Phase 5: Entry point refactoring (index.php uses RedirectController)
- Phase 6: Repository consolidation (AbstractDomainRepository, domain column fix)
- Phase 7: Admin panel refactoring (api.php from 1851 to 31 lines, modular controllers)

**Architecture Improvements:**
- God file `api.php` split into 9 focused API controllers
- Domain repositories share common logic via AbstractDomainRepository
- Request/Response abstractions for testability
- Clear separation of concerns following Clean Architecture
