<?php

declare(strict_types=1);

namespace App\Admin;

use App\Env;
use PDO;
use Throwable;

final class Auth
{
    private const SESSION_KEY_AUTH = 'admin_authed';
    private const SESSION_KEY_CSRF = 'csrf_token';
    private const SESSION_KEY_USER_ID = 'user_id';
    private const SESSION_KEY_USERNAME = 'username';
    private const SESSION_KEY_ROLE = 'user_role';

    public static function initSession(): void
    {
        if (session_status() === PHP_SESSION_ACTIVE) {
            return;
        }

        // Configure session save path for file-based sessions
        $sessionDriver = Env::getString('SESSION_DRIVER', 'file');
        if ($sessionDriver === 'file') {
            $sessionPath = __DIR__ . '/../../data/sessions';
            if (!is_dir($sessionPath)) {
                mkdir($sessionPath, 0700, true);
            }
            session_save_path($sessionPath);
        }

        $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');

        session_set_cookie_params([
            'lifetime' => 0,
            'path' => '/admin',
            'secure' => $secure,
            'httponly' => true,
            'samesite' => 'Strict',
        ]);

        ini_set('session.use_strict_mode', '1');
        session_start();
    }

    public static function ensureCsrfToken(): string
    {
        self::initSession();

        $token = $_SESSION[self::SESSION_KEY_CSRF] ?? '';
        if (is_string($token) && $token !== '') {
            return $token;
        }

        $token = rtrim(strtr(base64_encode(random_bytes(32)), '+/', '-_'), '=');
        $_SESSION[self::SESSION_KEY_CSRF] = $token;

        return $token;
    }

    /**
     * Alias untuk ensureCsrfToken() untuk backward compatibility
     */
    public static function generateCsrfToken(): string
    {
        return self::ensureCsrfToken();
    }

    public static function verifyCsrfToken(string $token): bool
    {
        self::initSession();

        $stored = $_SESSION[self::SESSION_KEY_CSRF] ?? '';
        if (!is_string($stored) || $stored === '') {
            return false;
        }

        return hash_equals($stored, $token);
    }

    public static function isAuthenticated(): bool
    {
        self::initSession();
        return ($_SESSION[self::SESSION_KEY_AUTH] ?? false) === true;
    }

    public static function requireAuth(): void
    {
        if (self::isAuthenticated()) {
            return;
        }

        header('Location: /admin/login.php', true, 302);
        exit;
    }

    public static function login(PDO $pdo, string $username, string $password): bool
    {
        self::initSession();

        $user = self::authenticateUser($pdo, $username, $password);

        if ($user !== null) {
            // Regenerate session ID to prevent session fixation
            session_regenerate_id(true);

            // Generate NEW CSRF token after login (security best practice)
            // Old token from login page should not be reused after authentication
            $_SESSION[self::SESSION_KEY_CSRF] = rtrim(strtr(base64_encode(random_bytes(32)), '+/', '-_'), '=');

            $_SESSION[self::SESSION_KEY_AUTH] = true;
            $_SESSION[self::SESSION_KEY_USER_ID] = $user['id'];
            $_SESSION[self::SESSION_KEY_USERNAME] = $user['username'];
            $_SESSION[self::SESSION_KEY_ROLE] = $user['role'];
            return true;
        }

        return false;
    }

    public static function logout(): void
    {
        self::initSession();

        $_SESSION = [];

        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            $sessionName = session_name();
            if ($sessionName !== false) {
                $samesite = $params['samesite'];
                /** @var 'Lax'|'Strict'|'None' $validSamesite */
                $validSamesite = in_array($samesite, ['Lax', 'Strict', 'None'], true) ? $samesite : 'Strict';

                setcookie($sessionName, '', [
                    'expires' => time() - 3600,
                    'path' => $params['path'],
                    'secure' => $params['secure'],
                    'httponly' => $params['httponly'],
                    'samesite' => $validSamesite,
                ]);
            }
        }

        session_destroy();
    }

    public static function getClientIp(): string
    {
        $ip = (string) ($_SERVER['REMOTE_ADDR'] ?? '');
        return $ip !== '' ? $ip : '0.0.0.0';
    }

    public static function getUserId(): ?int
    {
        self::initSession();
        $id = $_SESSION[self::SESSION_KEY_USER_ID] ?? null;
        return is_int($id) ? $id : null;
    }

    public static function getUsername(): ?string
    {
        self::initSession();
        $username = $_SESSION[self::SESSION_KEY_USERNAME] ?? null;
        return is_string($username) ? $username : null;
    }

    public static function getRole(): string
    {
        self::initSession();
        $role = $_SESSION[self::SESSION_KEY_ROLE] ?? 'user';
        return is_string($role) ? $role : 'user';
    }

    public static function isAdmin(): bool
    {
        return self::getRole() === 'admin';
    }

    public static function requireAdmin(): void
    {
        if (!self::isAdmin()) {
            http_response_code(403);
            echo 'Access denied. Admin role required.';
            exit;
        }
    }

    /**
     * Authenticate user with username and password
     * @return array{id:int,username:string,role:string}|null
     */
    private static function authenticateUser(PDO $pdo, string $username, string $password): ?array
    {
        try {
            $sql = 'SELECT id, username, password_hash, role FROM users WHERE username = :username AND is_active = 1 LIMIT 1';
            $stmt = $pdo->prepare($sql);
            $stmt->bindValue('username', $username, PDO::PARAM_STR);
            $stmt->execute();

            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!is_array($row)) {
                return null;
            }

            $hash = $row['password_hash'] ?? '';
            if (!is_string($hash) || !password_verify($password, $hash)) {
                return null;
            }

            return [
                'id' => (int) $row['id'],
                'username' => (string) $row['username'],
                'role' => (string) ($row['role'] ?? 'user'),
            ];
        } catch (Throwable $e) {
            error_log('auth_user_failed: ' . $e->getMessage());
            return null;
        }
    }

}
