<?php

declare(strict_types=1);

namespace App\Analytics;

use App\Cache;

/**
 * Async traffic logger for high-traffic scenarios
 *
 * Queues traffic data for background processing to avoid
 * blocking the redirect response. Uses APCu/file queue for
 * fast writes, with a worker processing the queue.
 */
final class AsyncTrafficLogger
{
    private const QUEUE_KEY = 'traffic_log_queue';
    private const QUEUE_TTL = 3600; // 1 hour
    private const MAX_QUEUE_SIZE = 10000;
    private const BATCH_SIZE = 100;

    /**
     * Queue traffic data for async processing
     *
     * @param array{
     *   shortlink_id?: int|null,
     *   shortlink_code: string,
     *   target_url: string,
     *   redirect_url?: string|null,
     *   ip_address: string,
     *   user_agent?: string|null,
     *   referer?: string|null,
     *   http_method?: string,
     *   http_protocol?: string|null,
     *   response_time_ms?: int|null
     * } $data
     * @return bool True if queued successfully
     */
    public function queue(array $data): bool
    {
        // Add timestamp for ordering
        $data['_queued_at'] = microtime(true);

        // Get current queue
        $queue = $this->getQueue();

        // Check queue size limit
        if (count($queue) >= self::MAX_QUEUE_SIZE) {
            // Queue full - try to process immediately or drop oldest
            error_log('async_traffic_queue_full: dropping oldest entries');
            $queue = array_slice($queue, -self::MAX_QUEUE_SIZE + 1);
        }

        // Add to queue
        $queue[] = $data;

        // Store queue
        return $this->saveQueue($queue);
    }

    /**
     * Process queued traffic data
     *
     * @param int $limit Maximum items to process
     * @return int Number of items processed
     */
    public function process(int $limit = self::BATCH_SIZE): int
    {
        $queue = $this->getQueue();

        if (empty($queue)) {
            return 0;
        }

        // Get items to process
        $toProcess = array_slice($queue, 0, $limit);
        $remaining = array_slice($queue, $limit);

        // Save remaining items first (to prevent loss on crash)
        $this->saveQueue($remaining);

        // Process items
        $logger = new TrafficLogger(\App\Db::pdo());
        $processed = 0;

        foreach ($toProcess as $data) {
            try {
                // Remove internal fields
                unset($data['_queued_at']);

                $logger->log($data);
                $processed++;
            } catch (\Throwable $e) {
                error_log('async_traffic_process_error: ' . $e->getMessage());
                // Re-queue failed item
                $this->requeue($data);
            }
        }

        return $processed;
    }

    /**
     * Get queue size
     */
    public function getQueueSize(): int
    {
        return count($this->getQueue());
    }

    /**
     * Clear the queue (use with caution)
     */
    public function clear(): void
    {
        $this->saveQueue([]);
    }

    /**
     * Get queue contents
     *
     * @return array<int, array<string, mixed>>
     */
    private function getQueue(): array
    {
        $queue = Cache::get(self::QUEUE_KEY);

        if (!is_array($queue)) {
            return [];
        }

        return $queue;
    }

    /**
     * Save queue
     *
     * @param array<int, array<string, mixed>> $queue
     */
    private function saveQueue(array $queue): bool
    {
        return Cache::set(self::QUEUE_KEY, $queue, self::QUEUE_TTL);
    }

    /**
     * Re-queue a failed item (adds to end)
     *
     * @param array<string, mixed> $data
     */
    private function requeue(array $data): void
    {
        // Add retry count
        $data['_retry_count'] = ($data['_retry_count'] ?? 0) + 1;

        // Max 3 retries
        if ($data['_retry_count'] > 3) {
            error_log('async_traffic_max_retries: ' . json_encode($data));
            return;
        }

        $queue = $this->getQueue();
        $queue[] = $data;
        $this->saveQueue($queue);
    }

    /**
     * Log traffic - either async or sync based on config
     *
     * @param array{
     *   shortlink_id?: int|null,
     *   shortlink_code: string,
     *   target_url: string,
     *   redirect_url?: string|null,
     *   ip_address: string,
     *   user_agent?: string|null,
     *   referer?: string|null,
     *   http_method?: string,
     *   http_protocol?: string|null,
     *   response_time_ms?: int|null
     * } $data
     * @param bool $async Force async mode
     */
    public static function log(array $data, bool $async = true): void
    {
        if ($async && \App\Env::getBool('TRAFFIC_LOG_ASYNC', true)) {
            $logger = new self();
            $logger->queue($data);
        } else {
            $logger = new TrafficLogger(\App\Db::pdo());
            $logger->log($data);
        }
    }
}
