<?php

declare(strict_types=1);

namespace App\Analytics;

use Detection\MobileDetect;

/**
 * Device, browser, and OS detection using Mobile Detect library.
 */
final class DeviceDetector
{
    private readonly MobileDetect $detector;

    public function __construct(?string $userAgent = null)
    {
        $this->detector = new MobileDetect();

        if ($userAgent !== null) {
            $this->detector->setUserAgent($userAgent);
        }
    }

    /**
     * Detect device, browser, and OS information.
     *
     * @return array{
     *   device_type: string,
     *   device_brand: string|null,
     *   device_model: string|null,
     *   browser_name: string|null,
     *   browser_version: string|null,
     *   os_name: string|null,
     *   os_version: string|null,
     *   platform: string|null,
     *   is_bot: bool,
     *   bot_name: string|null,
     *   user_agent: string|null
     * }
     */
    public function detect(): array
    {
        $userAgent = $this->detector->getUserAgent();
        $isBot = $this->isBot($userAgent ?? '');

        return [
            'device_type' => $this->getDeviceType(),
            'device_brand' => $this->getDeviceBrand(),
            'device_model' => $this->getDeviceModel(),
            'browser_name' => $this->getBrowserName(),
            'browser_version' => $this->getBrowserVersion(),
            'os_name' => $this->getOsName(),
            'os_version' => $this->getOsVersion(),
            'platform' => $this->getPlatform(),
            'is_bot' => $isBot,
            'bot_name' => $isBot ? $this->getBotName($userAgent ?? '') : null,
            'user_agent' => $userAgent,
        ];
    }

    private function getDeviceType(): string
    {
        if ($this->detector->isTablet()) {
            return 'tablet';
        }

        if ($this->detector->isMobile()) {
            return 'mobile';
        }

        // Check bot before desktop
        if ($this->isBot($this->detector->getUserAgent() ?? '')) {
            return 'bot';
        }

        return 'desktop';
    }

    private function getDeviceBrand(): ?string
    {
        $brands = ['Samsung', 'Apple', 'Huawei', 'Xiaomi', 'Oppo', 'Vivo', 'OnePlus', 'Google', 'LG', 'Sony', 'Motorola', 'Nokia', 'Asus', 'Lenovo'];

        foreach ($brands as $brand) {
            if ($this->detector->is($brand)) {
                return $brand;
            }
        }

        return null;
    }

    private function getDeviceModel(): null
    {
        // MobileDetect doesn't have reliable device model detection
        // This would require User-Agent parsing or dedicated device detection library
        return null;
    }

    private function getBrowserName(): ?string
    {
        $browsers = ['Chrome', 'Firefox', 'Safari', 'Edge', 'Opera', 'IE', 'Vivaldi', 'Brave'];

        foreach ($browsers as $browser) {
            if ($this->detector->is($browser)) {
                return $browser;
            }
        }

        return null;
    }

    private function getBrowserVersion(): ?string
    {
        $browserName = $this->getBrowserName();
        if ($browserName === null) {
            return null;
        }

        $version = $this->detector->version($browserName);
        return $version !== false ? (string) $version : null;
    }

    private function getOsName(): ?string
    {
        $osList = ['AndroidOS', 'iOS', 'WindowsPhoneOS', 'WindowsMobileOS', 'MacOS', 'Linux', 'Ubuntu', 'Windows'];

        foreach ($osList as $os) {
            if ($this->detector->is($os)) {
                return $os;
            }
        }

        return null;
    }

    private function getOsVersion(): ?string
    {
        $osName = $this->getOsName();
        if ($osName === null) {
            return null;
        }

        $version = $this->detector->version($osName);
        return $version !== false ? (string) $version : null;
    }

    private function getPlatform(): ?string
    {
        if ($this->detector->isiOS()) {
            return 'iOS';
        }

        if ($this->detector->isAndroidOS()) {
            return 'Android';
        }

        return null;
    }

    private function isBot(string $userAgent): bool
    {
        $botPatterns = [
            '/bot/i',
            '/crawler/i',
            '/spider/i',
            '/slurp/i',
            '/mediapartners/i',
            '/facebookexternalhit/i',
            '/twitterbot/i',
            '/linkedinbot/i',
            '/whatsapp/i',
            '/telegrambot/i',
            '/googlebot/i',
            '/bingbot/i',
            '/yandexbot/i',
            '/baiduspider/i',
            '/duckduckbot/i',
        ];

        foreach ($botPatterns as $pattern) {
            if (preg_match($pattern, $userAgent) === 1) {
                return true;
            }
        }

        return false;
    }

    private function getBotName(string $userAgent): string
    {
        $bots = [
            'Googlebot' => '/googlebot/i',
            'Bingbot' => '/bingbot/i',
            'Facebookbot' => '/facebookexternalhit/i',
            'Twitterbot' => '/twitterbot/i',
            'LinkedInBot' => '/linkedinbot/i',
            'WhatsApp' => '/whatsapp/i',
            'TelegramBot' => '/telegrambot/i',
            'Yandex' => '/yandexbot/i',
            'Baidu' => '/baiduspider/i',
            'DuckDuckBot' => '/duckduckbot/i',
            'Slurp' => '/slurp/i',
        ];

        foreach ($bots as $name => $pattern) {
            if (preg_match($pattern, $userAgent) === 1) {
                return $name;
            }
        }

        return 'Unknown Bot';
    }
}
