<?php

declare(strict_types=1);

namespace App\Analytics;

use GeoIp2\Database\Reader;
use GeoIp2\Exception\AddressNotFoundException;

/**
 * GeoIP detection using MaxMind GeoLite2 database.
 *
 * Required: Download GeoLite2-City.mmdb from MaxMind
 * Path: data/GeoLite2-City.mmdb
 */
final class GeoIpDetector
{
    private ?Reader $reader = null;

    public function __construct(
        private readonly string $dbPath = __DIR__ . '/../../data/GeoLite2-City.mmdb'
    ) {
    }

    /**
     * Detect geo information from IP address.
     *
     * @return array{
     *   success: bool,
     *   country_code: string|null,
     *   country_name: string|null,
     *   city: string|null,
     *   continent_code: string|null,
     *   latitude: float|null,
     *   longitude: float|null,
     *   error: string|null
     * }
     */
    public function detect(string $ip): array
    {
        try {
            if ($this->reader === null) {
                if (!file_exists($this->dbPath)) {
                    return $this->errorResponse('GeoLite2 database not found');
                }
                $this->reader = new Reader($this->dbPath);
            }

            $record = $this->reader->city($ip);

            return [
                'success' => true,
                'country_code' => $record->country->isoCode,
                'country_name' => $record->country->name,
                'city' => $record->city->name,
                'continent_code' => $record->continent->code,
                'latitude' => $record->location->latitude,
                'longitude' => $record->location->longitude,
                'error' => null,
            ];
        } catch (AddressNotFoundException $e) {
            return $this->errorResponse('IP not found in database');
        } catch (\Throwable $e) {
            return $this->errorResponse('GeoIP lookup failed: ' . $e->getMessage());
        }
    }

    /**
     * @return array{success: false, country_code: null, country_name: null, city: null, continent_code: null, latitude: null, longitude: null, error: string}
     */
    private function errorResponse(string $message): array
    {
        return [
            'success' => false,
            'country_code' => null,
            'country_name' => null,
            'city' => null,
            'continent_code' => null,
            'latitude' => null,
            'longitude' => null,
            'error' => $message,
        ];
    }

    public function __destruct()
    {
        if ($this->reader !== null) {
            $this->reader->close();
        }
    }
}
