<?php

declare(strict_types=1);

namespace App\Analytics;

use App\RiskScorer;
use PDO;

/**
 * Traffic analytics logger.
 *
 * Logs detailed visit data to traffic_analytics table with:
 * - GeoIP detection (country, city, lat/long)
 * - Device detection (type, brand, browser, OS)
 * - IP intelligence (VPN/proxy/Tor, risk score)
 * - HTTP request metadata
 */
final class TrafficLogger
{
    private readonly GeoIpDetector $geoIp;
    private readonly RiskScorer $riskScorer;

    public function __construct(
        private readonly PDO $pdo,
        ?GeoIpDetector $geoIp = null,
        ?RiskScorer $riskScorer = null
    ) {
        $this->geoIp = $geoIp ?? new GeoIpDetector();
        $this->riskScorer = $riskScorer ?? new RiskScorer();
    }

    /**
     * Log traffic visit.
     *
     * @param array{
     *   shortlink_id?: int|null,
     *   shortlink_code: string,
     *   target_url: string,
     *   redirect_url?: string|null,
     *   ip_address: string,
     *   user_agent?: string|null,
     *   referer?: string|null,
     *   http_method?: string,
     *   http_protocol?: string|null,
     *   response_time_ms?: int|null
     * } $data
     */
    public function log(array $data): void
    {
        $ip = $data['ip_address'];
        $userAgent = $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? null;

        // GeoIP detection
        $geo = $this->geoIp->detect($ip);

        // Device detection
        $deviceDetector = new DeviceDetector($userAgent);
        $device = $deviceDetector->detect();

        // IP intelligence / risk scoring
        $risk = $this->riskScorer->assess($ip);

        // Convert IP to binary for storage (supports both IPv4 and IPv6)
        $ipBin = @inet_pton($ip);
        if ($ipBin === false) {
            // Invalid IP, use placeholder
            $ipBin = inet_pton('0.0.0.0');
        }

        $stmt = $this->pdo->prepare('
            INSERT INTO traffic_analytics (
                shortlink_id,
                shortlink_code,
                target_url,
                redirect_url,
                ip_bin,
                country_code,
                country_name,
                city,
                continent_code,
                latitude,
                longitude,
                asn,
                asn_org,
                isp,
                is_vpn,
                is_proxy,
                is_tor,
                is_datacenter,
                risk_score,
                blackbox_signal,
                device_type,
                device_brand,
                device_model,
                browser_name,
                browser_version,
                os_name,
                os_version,
                platform,
                user_agent,
                is_bot,
                bot_name,
                referer,
                http_method,
                http_protocol,
                response_time_ms
            ) VALUES (
                :shortlink_id,
                :shortlink_code,
                :target_url,
                :redirect_url,
                :ip_bin,
                :country_code,
                :country_name,
                :city,
                :continent_code,
                :latitude,
                :longitude,
                :asn,
                :asn_org,
                :isp,
                :is_vpn,
                :is_proxy,
                :is_tor,
                :is_datacenter,
                :risk_score,
                :blackbox_signal,
                :device_type,
                :device_brand,
                :device_model,
                :browser_name,
                :browser_version,
                :os_name,
                :os_version,
                :platform,
                :user_agent,
                :is_bot,
                :bot_name,
                :referer,
                :http_method,
                :http_protocol,
                :response_time_ms
            )
        ');

        $stmt->bindValue('shortlink_id', $data['shortlink_id'] ?? null, PDO::PARAM_INT);
        $stmt->bindValue('shortlink_code', $data['shortlink_code'], PDO::PARAM_STR);
        $stmt->bindValue('target_url', $data['target_url'], PDO::PARAM_STR);
        $stmt->bindValue('redirect_url', $data['redirect_url'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue('ip_bin', $ipBin, PDO::PARAM_LOB); // Binary data
        $stmt->bindValue('country_code', $geo['country_code'], PDO::PARAM_STR);
        $stmt->bindValue('country_name', $geo['country_name'], PDO::PARAM_STR);
        $stmt->bindValue('city', $geo['city'], PDO::PARAM_STR);
        $stmt->bindValue('continent_code', $geo['continent_code'], PDO::PARAM_STR);
        $stmt->bindValue('latitude', $geo['latitude']);
        $stmt->bindValue('longitude', $geo['longitude']);
        $stmt->bindValue('asn', $risk['asn'], PDO::PARAM_INT);
        $stmt->bindValue('asn_org', null, PDO::PARAM_STR);
        $stmt->bindValue('isp', null, PDO::PARAM_STR);
        $stmt->bindValue('is_vpn', $risk['is_vpn'] ? 1 : 0, PDO::PARAM_INT);
        $stmt->bindValue('is_proxy', $risk['is_proxy'] ? 1 : 0, PDO::PARAM_INT);
        $stmt->bindValue('is_tor', $risk['is_tor'] ? 1 : 0, PDO::PARAM_INT);
        $stmt->bindValue('is_datacenter', $risk['is_datacenter'] ? 1 : 0, PDO::PARAM_INT);
        $stmt->bindValue('risk_score', min(100, max(0, (int) $risk['risk_score'])), PDO::PARAM_INT);
        $stmt->bindValue('blackbox_signal', $risk['blackbox_signal'], PDO::PARAM_STR);
        $stmt->bindValue('device_type', $device['device_type'], PDO::PARAM_STR);
        $stmt->bindValue('device_brand', $device['device_brand'], PDO::PARAM_STR);
        $stmt->bindValue('device_model', $device['device_model'], PDO::PARAM_STR);
        $stmt->bindValue('browser_name', $device['browser_name'], PDO::PARAM_STR);
        $stmt->bindValue('browser_version', $device['browser_version'], PDO::PARAM_STR);
        $stmt->bindValue('os_name', $device['os_name'], PDO::PARAM_STR);
        $stmt->bindValue('os_version', $device['os_version'], PDO::PARAM_STR);
        $stmt->bindValue('platform', $device['platform'], PDO::PARAM_STR);
        $stmt->bindValue('user_agent', $device['user_agent'], PDO::PARAM_STR);
        $stmt->bindValue('is_bot', $device['is_bot'] ? 1 : 0, PDO::PARAM_INT);
        $stmt->bindValue('bot_name', $device['bot_name'], PDO::PARAM_STR);
        $stmt->bindValue('referer', $data['referer'] ?? $_SERVER['HTTP_REFERER'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue('http_method', $data['http_method'] ?? $_SERVER['REQUEST_METHOD'] ?? 'GET', PDO::PARAM_STR);
        $stmt->bindValue('http_protocol', $data['http_protocol'] ?? $_SERVER['SERVER_PROTOCOL'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue('response_time_ms', $data['response_time_ms'] ?? null, PDO::PARAM_INT);

        $stmt->execute();
    }

    /**
     * Get analytics summary for a shortlink.
     *
     * Optimized: Single query with JSON aggregation instead of 4 separate queries.
     * Falls back to daily summary table for large datasets.
     *
     * @return array{
     *   total_visits: int,
     *   unique_ips: int,
     *   bot_visits: int,
     *   mobile_visits: int,
     *   desktop_visits: int,
     *   tablet_visits: int,
     *   risky_visits: int,
     *   top_countries: array<string, int>,
     *   top_devices: array<string, int>,
     *   top_browsers: array<string, int>
     * }
     */
    public function getSummary(string $shortlinkCode, int $days = 30): array
    {
        // Try to use pre-aggregated daily summary for better performance
        $useDailySummary = $days >= 7;

        if ($useDailySummary) {
            $result = $this->getSummaryFromDaily($shortlinkCode, $days);
            if ($result !== null) {
                return $result;
            }
        }

        // Single optimized query with all metrics
        $stmt = $this->pdo->prepare('
            SELECT
                COUNT(*) as total_visits,
                COUNT(DISTINCT ip_bin) as unique_ips,
                SUM(is_bot) as bot_visits,
                SUM(CASE WHEN device_type = "mobile" THEN 1 ELSE 0 END) as mobile_visits,
                SUM(CASE WHEN device_type = "desktop" THEN 1 ELSE 0 END) as desktop_visits,
                SUM(CASE WHEN device_type = "tablet" THEN 1 ELSE 0 END) as tablet_visits,
                SUM(CASE WHEN risk_score >= 75 THEN 1 ELSE 0 END) as risky_visits,
                (
                    SELECT JSON_OBJECTAGG(country_code, cnt)
                    FROM (
                        SELECT country_code, COUNT(*) as cnt
                        FROM traffic_analytics t2
                        WHERE t2.shortlink_code = t1.shortlink_code
                          AND t2.visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                          AND t2.country_code IS NOT NULL
                        GROUP BY country_code
                        ORDER BY cnt DESC
                        LIMIT 10
                    ) sub
                ) as top_countries_json,
                (
                    SELECT JSON_OBJECTAGG(device_brand, cnt)
                    FROM (
                        SELECT device_brand, COUNT(*) as cnt
                        FROM traffic_analytics t3
                        WHERE t3.shortlink_code = t1.shortlink_code
                          AND t3.visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                          AND t3.device_brand IS NOT NULL
                        GROUP BY device_brand
                        ORDER BY cnt DESC
                        LIMIT 10
                    ) sub
                ) as top_devices_json,
                (
                    SELECT JSON_OBJECTAGG(browser_name, cnt)
                    FROM (
                        SELECT browser_name, COUNT(*) as cnt
                        FROM traffic_analytics t4
                        WHERE t4.shortlink_code = t1.shortlink_code
                          AND t4.visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                          AND t4.browser_name IS NOT NULL
                        GROUP BY browser_name
                        ORDER BY cnt DESC
                        LIMIT 10
                    ) sub
                ) as top_browsers_json
            FROM traffic_analytics t1
            WHERE shortlink_code = ?
              AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
        ');

        $stmt->execute([$days, $days, $days, $shortlinkCode, $days]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row === false || $row['total_visits'] === null) {
            return $this->emptyResult();
        }

        return [
            'total_visits' => (int) ($row['total_visits'] ?? 0),
            'unique_ips' => (int) ($row['unique_ips'] ?? 0),
            'bot_visits' => (int) ($row['bot_visits'] ?? 0),
            'mobile_visits' => (int) ($row['mobile_visits'] ?? 0),
            'desktop_visits' => (int) ($row['desktop_visits'] ?? 0),
            'tablet_visits' => (int) ($row['tablet_visits'] ?? 0),
            'risky_visits' => (int) ($row['risky_visits'] ?? 0),
            'top_countries' => $this->parseJsonAgg($row['top_countries_json'] ?? null),
            'top_devices' => $this->parseJsonAgg($row['top_devices_json'] ?? null),
            'top_browsers' => $this->parseJsonAgg($row['top_browsers_json'] ?? null),
        ];
    }

    /**
     * Get summary from pre-aggregated daily table (faster for large datasets)
     *
     * @return array<string, mixed>|null
     */
    private function getSummaryFromDaily(string $shortlinkCode, int $days): ?array
    {
        $stmt = $this->pdo->prepare('
            SELECT
                SUM(total_visits) as total_visits,
                SUM(unique_ips) as unique_ips,
                SUM(bot_visits) as bot_visits,
                SUM(mobile_visits) as mobile_visits,
                SUM(desktop_visits) as desktop_visits,
                SUM(tablet_visits) as tablet_visits,
                SUM(risky_visits) as risky_visits,
                GROUP_CONCAT(top_countries) as countries_concat,
                GROUP_CONCAT(top_devices) as devices_concat,
                GROUP_CONCAT(top_browsers) as browsers_concat
            FROM traffic_summary_daily
            WHERE shortlink_code = ?
              AND summary_date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
        ');

        $stmt->execute([$shortlinkCode, $days]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row === false || $row['total_visits'] === null || (int) $row['total_visits'] === 0) {
            return null; // Fall back to detailed query
        }

        return [
            'total_visits' => (int) ($row['total_visits'] ?? 0),
            'unique_ips' => (int) ($row['unique_ips'] ?? 0),
            'bot_visits' => (int) ($row['bot_visits'] ?? 0),
            'mobile_visits' => (int) ($row['mobile_visits'] ?? 0),
            'desktop_visits' => (int) ($row['desktop_visits'] ?? 0),
            'tablet_visits' => (int) ($row['tablet_visits'] ?? 0),
            'risky_visits' => (int) ($row['risky_visits'] ?? 0),
            'top_countries' => $this->mergeJsonStrings($row['countries_concat'] ?? ''),
            'top_devices' => $this->mergeJsonStrings($row['devices_concat'] ?? ''),
            'top_browsers' => $this->mergeJsonStrings($row['browsers_concat'] ?? ''),
        ];
    }

    /**
     * Parse JSON aggregation result
     *
     * @return array<string, int>
     */
    private function parseJsonAgg(?string $json): array
    {
        if ($json === null || $json === '') {
            return [];
        }

        $data = json_decode($json, true);
        if (!is_array($data)) {
            return [];
        }

        // Ensure all values are integers
        $result = [];
        foreach ($data as $key => $value) {
            $result[(string) $key] = (int) $value;
        }

        return $result;
    }

    /**
     * Merge multiple JSON strings and aggregate counts
     *
     * @return array<string, int>
     */
    private function mergeJsonStrings(string $concat): array
    {
        if ($concat === '') {
            return [];
        }

        $merged = [];
        $jsonStrings = explode(',', $concat);

        foreach ($jsonStrings as $json) {
            $data = json_decode($json, true);
            if (is_array($data)) {
                foreach ($data as $key => $value) {
                    $key = (string) $key;
                    $merged[$key] = ($merged[$key] ?? 0) + (int) $value;
                }
            }
        }

        // Sort by count descending and limit to top 10
        arsort($merged);
        return array_slice($merged, 0, 10, true);
    }

    /**
     * Return empty result structure
     *
     * @return array<string, mixed>
     */
    private function emptyResult(): array
    {
        return [
            'total_visits' => 0,
            'unique_ips' => 0,
            'bot_visits' => 0,
            'mobile_visits' => 0,
            'desktop_visits' => 0,
            'tablet_visits' => 0,
            'risky_visits' => 0,
            'top_countries' => [],
            'top_devices' => [],
            'top_browsers' => [],
        ];
    }
}
