<?php

declare(strict_types=1);

namespace App;

/**
 * Base64url encoding/decoding (URL-safe, no padding)
 *
 * Converts standard Base64:
 * - Replace '+' with '-'
 * - Replace '/' with '_'
 * - Remove padding '='
 *
 * @see https://tools.ietf.org/html/rfc4648#section-5
 */
final class Base64Url
{
    /**
     * Encode data to Base64url
     *
     * @param string $data Raw data to encode
     * @return string Base64url encoded string
     */
    public static function encode(string $data): string
    {
        $base64 = base64_encode($data);

        // Convert to URL-safe variant
        $base64url = strtr($base64, '+/', '-_');

        // Remove padding
        return rtrim($base64url, '=');
    }

    /**
     * Decode Base64url data
     *
     * @param string $data Base64url encoded string
     * @return string|false Decoded data, or false on failure
     */
    public static function decode(string $data): string|false
    {
        // Convert from URL-safe variant
        $base64 = strtr($data, '-_', '+/');

        // Add padding if needed
        $remainder = strlen($base64) % 4;
        if ($remainder > 0) {
            $base64 .= str_repeat('=', 4 - $remainder);
        }

        return base64_decode($base64, true);
    }

    /**
     * Encode binary data (hex to base64url)
     *
     * @param string $hex Hexadecimal string
     * @return string Base64url encoded string
     */
    public static function encodeHex(string $hex): string
    {
        $binary = hex2bin($hex);
        if ($binary === false) {
            return '';
        }

        return self::encode($binary);
    }

    /**
     * Decode to hex (base64url to hex)
     *
     * @param string $data Base64url encoded string
     * @return string Hexadecimal string
     */
    public static function decodeToHex(string $data): string
    {
        $binary = self::decode($data);
        if ($binary === false) {
            return '';
        }

        return bin2hex($binary);
    }

    /**
     * Generate random Base64url token
     *
     * @param int $bytes Number of random bytes (default 32)
     * @return string Base64url encoded random token
     */
    public static function randomToken(int $bytes = 32): string
    {
        try {
            $randomBytes = random_bytes($bytes);
            return self::encode($randomBytes);
        } catch (\Throwable $e) {
            return '';
        }
    }
}
