<?php

declare(strict_types=1);

namespace App;

/**
 * Simple cache implementation with APCu support
 * Falls back to file-based cache if APCu not available
 */
final class Cache
{
    private static bool $initialized = false;
    private static bool $apcu = false;
    private static string $cacheDir = '';

    public static function init(): void
    {
        if (self::$initialized) {
            return;
        }

        self::$apcu = extension_loaded('apcu') && apcu_enabled();
        self::$cacheDir = __DIR__ . '/../data/cache';

        if (!self::$apcu && !is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }

        self::$initialized = true;
    }

    /**
     * Get value from cache
     * @return mixed|null
     */
    public static function get(string $key): mixed
    {
        self::init();

        if (self::$apcu) {
            $value = apcu_fetch($key, $success);
            return $success ? $value : null;
        }

        // File-based fallback
        $file = self::$cacheDir . '/' . self::sanitizeKey($key) . '.cache';
        if (!file_exists($file)) {
            return null;
        }

        $content = file_get_contents($file);
        if ($content === false) {
            return null;
        }

        $data = json_decode($content, true);
        if (!is_array($data)) {
            return null;
        }

        // Check expiry
        if (isset($data['expire']) && $data['expire'] < time()) {
            @unlink($file);
            return null;
        }

        return $data['value'] ?? null;
    }

    /**
     * Set value in cache with TTL
     */
    public static function set(string $key, mixed $value, int $ttl = 300): bool
    {
        self::init();

        if (self::$apcu) {
            return apcu_store($key, $value, $ttl);
        }

        // File-based fallback
        $file = self::$cacheDir . '/' . self::sanitizeKey($key) . '.cache';
        $data = [
            'value' => $value,
            'expire' => time() + $ttl,
        ];

        return file_put_contents($file, json_encode($data, JSON_THROW_ON_ERROR)) !== false;
    }

    /**
     * Delete value from cache
     */
    public static function delete(string $key): bool
    {
        self::init();

        if (self::$apcu) {
            return apcu_delete($key);
        }

        // File-based fallback
        $file = self::$cacheDir . '/' . self::sanitizeKey($key) . '.cache';
        if (file_exists($file)) {
            return @unlink($file);
        }

        return true;
    }

    /**
     * Fetch value from cache (APCu-compatible signature)
     *
     * @param string $key Cache key
     * @param bool|null $success Output parameter indicating success
     * @return mixed Cached value or false
     */
    public static function fetch(string $key, ?bool &$success = null): mixed
    {
        $value = self::get($key);

        if ($value !== null) {
            $success = true;
            return $value;
        }

        $success = false;
        return false;
    }

    /**
     * Store value in cache (APCu-compatible alias)
     *
     * @param string $key Cache key
     * @param mixed $value Value to store
     * @param int $ttl Time to live in seconds
     * @return bool Success status
     */
    public static function store(string $key, mixed $value, int $ttl = 300): bool
    {
        return self::set($key, $value, $ttl);
    }

    /**
     * Check if key exists in cache
     *
     * @param string $key Cache key
     * @return bool True if exists and not expired
     */
    public static function exists(string $key): bool
    {
        return self::get($key) !== null;
    }

    /**
     * Increment counter atomically (for metrics)
     *
     * @param string $key Counter key
     * @param int $step Increment step (default 1)
     * @param int $ttl Time to live in seconds
     * @return int New counter value
     */
    public static function increment(string $key, int $step = 1, int $ttl = 3600): int
    {
        self::init();

        if (self::$apcu) {
            // Use atomic increment - creates key with initial value if not exists
            $success = false;
            $newValue = apcu_inc($key, $step, $success, $ttl);

            if ($success && $newValue !== false) {
                return $newValue;
            }

            // Key doesn't exist, create it atomically
            if (apcu_add($key, $step, $ttl)) {
                return $step;
            }

            // Race: another process created it, try increment again
            $newValue = apcu_inc($key, $step, $success, $ttl);
            return $success && $newValue !== false ? $newValue : $step;
        }

        // File-based fallback (not atomic, but acceptable for low-traffic)
        $current = (int) self::get($key);
        $newValue = $current + $step;
        self::set($key, $newValue, $ttl);
        return $newValue;
    }

    /**
     * Decrement counter atomically
     *
     * @param string $key Counter key
     * @param int $step Decrement step (default 1)
     * @param int $ttl Time to live in seconds
     * @return int New counter value (minimum 0)
     */
    public static function decrement(string $key, int $step = 1, int $ttl = 3600): int
    {
        self::init();

        if (self::$apcu) {
            $success = false;
            $newValue = apcu_dec($key, $step, $success, $ttl);

            if ($success && $newValue !== false) {
                return max(0, $newValue);
            }

            return 0;
        }

        // File-based fallback
        $current = (int) self::get($key);
        $newValue = max(0, $current - $step);
        self::set($key, $newValue, $ttl);
        return $newValue;
    }

    /**
     * Delete all cache entries matching pattern
     * Pattern: "shortlink:*:demo" will delete all keys with that pattern
     */
    public static function deletePattern(string $pattern): int
    {
        self::init();

        $deleted = 0;

        if (self::$apcu) {
            // APCu: use iterator to find matching keys
            $regexPattern = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
            $iterator = new \APCUIterator($regexPattern);
            foreach ($iterator as $entry) {
                if (is_array($entry) && isset($entry['key']) && is_string($entry['key'])) {
                    if (apcu_delete($entry['key'])) {
                        $deleted++;
                    }
                }
            }
            return $deleted;
        }

        // File-based fallback: scan all cache files and delete matching
        $files = glob(self::$cacheDir . '/*.cache');
        if ($files === false) {
            return $deleted;
        }
        foreach ($files as $file) {
            $basename = basename($file, '.cache');
            // Try to reverse the sanitization to match pattern
            $regexPattern = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
            // For file-based, we need to check against the sanitized pattern
            $filePattern = self::sanitizeKey($pattern);
            $filePattern = str_replace('_', '*', $filePattern); // wildcards become *

            if (fnmatch('*' . $filePattern . '*', $basename) || fnmatch($filePattern, $basename)) {
                if (@unlink($file)) {
                    $deleted++;
                }
            }
        }

        return $deleted;
    }

    /**
     * Clear all cache
     */
    public static function clear(): bool
    {
        self::init();

        if (self::$apcu) {
            return apcu_clear_cache();
        }

        // File-based fallback: delete all cache files
        $files = glob(self::$cacheDir . '/*.cache');
        if ($files !== false) {
            foreach ($files as $file) {
                @unlink($file);
            }
        }

        return true;
    }

    /**
     * Sanitize cache key for filename
     */
    private static function sanitizeKey(string $key): string
    {
        $result = preg_replace('/[^a-zA-Z0-9_-]/', '_', $key);
        return $result !== null ? $result : '';
    }

    /**
     * Generate cache key for shortlink per IP
     */
    public static function shortlinkKey(string $ip, string $code): string
    {
        $ipHash = hash('sha256', $ip);
        return "shortlink:{$ipHash}:{$code}";
    }

    /**
     * Invalidate all cache entries for a specific shortlink code
     * This will delete all cached entries regardless of IP for this shortlink
     */
    public static function invalidateShortlink(string $code): int
    {
        self::init();

        $deleted = 0;

        if (self::$apcu) {
            // APCu: iterate and match pattern
            $prefix = "shortlink:";
            $suffix = ":{$code}";
            $iterator = new \APCUIterator();
            foreach ($iterator as $entry) {
                if (is_array($entry) && isset($entry['key']) && is_string($entry['key'])) {
                    $key = $entry['key'];
                    if (strpos($key, $prefix) === 0 && substr($key, -strlen($suffix)) === $suffix) {
                        if (apcu_delete($key)) {
                            $deleted++;
                        }
                    }
                }
            }
            return $deleted;
        }

        // File-based: scan all files and check content
        $files = glob(self::$cacheDir . '/*.cache');
        if ($files === false) {
            return $deleted;
        }
        $sanitizedCode = self::sanitizeKey($code);

        foreach ($files as $file) {
            $basename = basename($file, '.cache');
            // Check if filename ends with sanitized code
            if (substr($basename, -strlen($sanitizedCode)) === $sanitizedCode) {
                if (@unlink($file)) {
                    $deleted++;
                }
            }
        }

        return $deleted;
    }
}
