<?php

declare(strict_types=1);

namespace App;

/**
 * Cloudflare API Integration for DNS Management
 *
 * Uses Cloudflare API v4 with API Token authentication
 *
 * Requirements:
 * - Cloudflare account
 * - API Token with Zone:Edit permissions (My Profile > API Tokens)
 *
 * Features:
 * - Add DNS record (A, CNAME, TXT, etc.)
 * - Add wildcard DNS record
 * - List DNS records
 * - Update DNS record
 * - Delete DNS record
 * - Purge cache for specific URLs
 */
final class CloudflareApi
{
    private string $baseUrl = 'https://api.cloudflare.com/client/v4';
    private string $apiToken;
    private ?string $zoneId;

    /**
     * @param string $apiToken Cloudflare API token
     * @param string|null $zoneId Zone ID (optional, can be fetched automatically)
     */
    public function __construct(string $apiToken, ?string $zoneId = null)
    {
        $this->apiToken = $apiToken;
        $this->zoneId = $zoneId;
    }

    /**
     * Get zone ID by domain name
     *
     * @param string $domain Domain name (e.g., example.com)
     * @return array{success: bool, message: string, zone_id?: string}
     */
    public function getZoneId(string $domain): array
    {
        $result = $this->apiRequest('GET', '/zones', ['name' => $domain]);

        if (!$result['success']) {
            return $result;
        }

        $zones = $result['data']['result'] ?? [];
        if (empty($zones)) {
            return [
                'success' => false,
                'message' => 'Zone not found for domain: ' . $domain,
            ];
        }

        $zoneId = (string) ($zones[0]['id'] ?? '');
        if ($zoneId === '') {
            return [
                'success' => false,
                'message' => 'Invalid zone ID',
            ];
        }

        $this->zoneId = $zoneId;

        return [
            'success' => true,
            'message' => 'Zone ID retrieved',
            'zone_id' => $zoneId,
        ];
    }

    /**
     * Add DNS record
     *
     * @param string $type Record type (A, AAAA, CNAME, TXT, MX, etc.)
     * @param string $name Record name (e.g., @ for root, www, *, *.api)
     * @param string $content Record content (IP address, domain, etc.)
     * @param int $ttl TTL in seconds (1 = automatic)
     * @param bool $proxied Enable Cloudflare proxy (orange cloud)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function addDnsRecord(string $type, string $name, string $content, int $ttl = 1, bool $proxied = false): array
    {
        if ($this->zoneId === null) {
            return [
                'success' => false,
                'message' => 'Zone ID not set. Call getZoneId() first.',
            ];
        }

        $payload = [
            'type' => strtoupper($type),
            'name' => $name,
            'content' => $content,
            'ttl' => $ttl,
            'proxied' => $proxied,
        ];

        return $this->apiRequest('POST', '/zones/' . $this->zoneId . '/dns_records', [], $payload);
    }

    /**
     * List DNS records
     *
     * @param string|null $type Filter by record type (optional)
     * @param string|null $name Filter by record name (optional)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function listDnsRecords(?string $type = null, ?string $name = null): array
    {
        if ($this->zoneId === null) {
            return [
                'success' => false,
                'message' => 'Zone ID not set. Call getZoneId() first.',
            ];
        }

        $params = [];
        if ($type !== null) {
            $params['type'] = strtoupper($type);
        }
        if ($name !== null) {
            $params['name'] = $name;
        }

        return $this->apiRequest('GET', '/zones/' . $this->zoneId . '/dns_records', $params);
    }

    /**
     * Update DNS record
     *
     * @param string $recordId DNS record ID
     * @param string $type Record type
     * @param string $name Record name
     * @param string $content Record content
     * @param int $ttl TTL in seconds
     * @param bool $proxied Enable Cloudflare proxy
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function updateDnsRecord(string $recordId, string $type, string $name, string $content, int $ttl = 1, bool $proxied = false): array
    {
        if ($this->zoneId === null) {
            return [
                'success' => false,
                'message' => 'Zone ID not set. Call getZoneId() first.',
            ];
        }

        $payload = [
            'type' => strtoupper($type),
            'name' => $name,
            'content' => $content,
            'ttl' => $ttl,
            'proxied' => $proxied,
        ];

        return $this->apiRequest('PUT', '/zones/' . $this->zoneId . '/dns_records/' . $recordId, [], $payload);
    }

    /**
     * Delete DNS record
     *
     * @param string $recordId DNS record ID
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function deleteDnsRecord(string $recordId): array
    {
        if ($this->zoneId === null) {
            return [
                'success' => false,
                'message' => 'Zone ID not set. Call getZoneId() first.',
            ];
        }

        return $this->apiRequest('DELETE', '/zones/' . $this->zoneId . '/dns_records/' . $recordId);
    }

    /**
     * Purge cache for specific URLs
     *
     * @param list<string> $urls List of URLs to purge
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function purgeCache(array $urls): array
    {
        if ($this->zoneId === null) {
            return [
                'success' => false,
                'message' => 'Zone ID not set. Call getZoneId() first.',
            ];
        }

        return $this->apiRequest('POST', '/zones/' . $this->zoneId . '/purge_cache', [], ['files' => $urls]);
    }

    /**
     * Make Cloudflare API request
     *
     * @param string $method HTTP method (GET, POST, PUT, DELETE)
     * @param string $endpoint API endpoint (e.g., /zones)
     * @param array<string, mixed> $params Query parameters
     * @param array<string, mixed>|null $payload Request body (for POST/PUT)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    private function apiRequest(string $method, string $endpoint, array $params = [], ?array $payload = null): array
    {
        $url = $this->baseUrl . $endpoint;

        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $this->apiToken,
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        if ($payload !== null && in_array($method, ['POST', 'PUT', 'PATCH'], true)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload, JSON_THROW_ON_ERROR));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($response === false) {
            return [
                'success' => false,
                'message' => 'cURL error: ' . $error,
            ];
        }

        $data = json_decode($response, true);
        if (!is_array($data)) {
            return [
                'success' => false,
                'message' => 'Invalid JSON response (HTTP ' . $httpCode . ')',
            ];
        }

        // Cloudflare API response format:
        // {"success": true/false, "errors": [], "messages": [], "result": {...}}
        $success = (bool) ($data['success'] ?? false);
        $errors = $data['errors'] ?? [];
        $messages = $data['messages'] ?? [];

        if ($success) {
            return [
                'success' => true,
                'message' => !empty($messages) ? implode(', ', array_column($messages, 'message')) : 'Success',
                'data' => $data,
            ];
        }

        $errorMsg = !empty($errors) ? implode(', ', array_column($errors, 'message')) : 'Unknown error';

        return [
            'success' => false,
            'message' => $errorMsg . ' (HTTP ' . $httpCode . ')',
        ];
    }
}
