<?php

declare(strict_types=1);

namespace App;

/**
 * cPanel API Integration for Domain/Subdomain Management
 *
 * Uses cPanel UAPI (Unified API) with API Token authentication
 *
 * Requirements:
 * - cPanel WHM/UAPI access
 * - API Token from cPanel (Security > Manage API Tokens)
 *
 * Features:
 * - Add addon domain
 * - Add subdomain
 * - Add wildcard DNS record
 * - List domains
 * - Delete domain
 */
final class CpanelApi
{
    private string $baseUrl;
    private string $username;
    private string $apiToken;

    /**
     * @param string $host cPanel hostname (e.g., cpanel.example.com)
     * @param string $username cPanel username
     * @param string $apiToken cPanel API token
     */
    public function __construct(string $host, string $username, string $apiToken)
    {
        $this->baseUrl = 'https://' . rtrim($host, '/') . ':2083/execute';
        $this->username = $username;
        $this->apiToken = $apiToken;
    }

    /**
     * Add addon domain (parked domain with own document root)
     *
     * @param string $domain Domain name (e.g., short.example.com)
     * @param string $subdomain Subdomain name (e.g., short)
     * @param string $dir Document root (e.g., public_html/short)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function addAddonDomain(string $domain, string $subdomain, string $dir): array
    {
        $params = [
            'domain' => $domain,
            'subdomain' => $subdomain,
            'dir' => $dir,
        ];

        return $this->callUapi('AddonDomain', 'addaddondomain', $params);
    }

    /**
     * Add subdomain under main domain
     *
     * @param string $domain Subdomain name (e.g., api)
     * @param string $rootdomain Main domain (e.g., example.com)
     * @param string $dir Document root (e.g., public_html/api)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function addSubdomain(string $domain, string $rootdomain, string $dir): array
    {
        $params = [
            'domain' => $domain,
            'rootdomain' => $rootdomain,
            'dir' => $dir,
        ];

        return $this->callUapi('SubDomain', 'addsubdomain', $params);
    }

    /**
     * Add DNS zone record (for wildcard support)
     *
     * @param string $domain Zone domain (e.g., example.com)
     * @param string $name Record name (e.g., *.short or * for wildcard)
     * @param string $type Record type (A, CNAME, etc.)
     * @param string $address Target (IP address or domain)
     * @param int $ttl TTL in seconds (default: 14400 = 4 hours)
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function addDnsRecord(string $domain, string $name, string $type, string $address, int $ttl = 14400): array
    {
        $params = [
            'domain' => $domain,
            'name' => $name,
            'type' => $type,
            'address' => $address,
            'ttl' => $ttl,
        ];

        return $this->callUapi('DNS', 'add_zone_record', $params);
    }

    /**
     * List addon domains
     *
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function listAddonDomains(): array
    {
        return $this->callUapi('AddonDomain', 'listaddondomains', []);
    }

    /**
     * Delete addon domain
     *
     * @param string $domain Domain to delete
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    public function deleteAddonDomain(string $domain): array
    {
        return $this->callUapi('AddonDomain', 'deladdondomain', ['domain' => $domain]);
    }

    /**
     * Call cPanel UAPI endpoint
     *
     * @param string $module UAPI module (e.g., DNS, SubDomain)
     * @param string $function UAPI function (e.g., add_zone_record)
     * @param array<string, mixed> $params Function parameters
     * @return array{success: bool, message: string, data?: array<string, mixed>}
     */
    private function callUapi(string $module, string $function, array $params): array
    {
        $url = $this->baseUrl . '/' . rawurlencode($module) . '/' . rawurlencode($function);

        if (!empty($params)) {
            $query = http_build_query($params);
            $url .= '?' . $query;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: cpanel ' . $this->username . ':' . $this->apiToken,
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($response === false) {
            return [
                'success' => false,
                'message' => 'cURL error: ' . $error,
            ];
        }

        if ($httpCode !== 200) {
            return [
                'success' => false,
                'message' => 'HTTP error: ' . $httpCode,
            ];
        }

        $data = json_decode($response, true);
        if (!is_array($data)) {
            return [
                'success' => false,
                'message' => 'Invalid JSON response',
            ];
        }

        // cPanel UAPI response format:
        // {"status": 1, "errors": null, "data": {...}, "metadata": {...}}
        $status = (int) ($data['status'] ?? 0);
        $errors = $data['errors'] ?? null;
        $resultData = $data['data'] ?? [];

        if ($status === 1 && $errors === null) {
            return [
                'success' => true,
                'message' => 'Success',
                'data' => $resultData,
            ];
        }

        $errorMsg = is_array($errors) && !empty($errors) ? implode(', ', $errors) : 'Unknown error';

        return [
            'success' => false,
            'message' => $errorMsg,
        ];
    }
}
