<?php

declare(strict_types=1);

namespace App\Database;

use PDO;
use RuntimeException;

/**
 * Connection pool untuk PDO dengan singleton pattern per DSN
 */
final class ConnectionPool
{
    /**
     * @var array<string, self>
     */
    private static array $instances = [];

    private PDO $connection;

    private function __construct(
        private readonly string $dsn,
        private readonly string $user,
        private readonly string $pass
    ) {
        $this->connection = $this->createConnection();
    }

    /**
     * Get singleton instance untuk DSN tertentu
     */
    public static function getInstance(string $dsn, string $user, string $pass): self
    {
        $key = md5($dsn . $user);

        if (!isset(self::$instances[$key])) {
            self::$instances[$key] = new self($dsn, $user, $pass);
        }

        return self::$instances[$key];
    }

    /**
     * Get PDO connection
     */
    public function getConnection(): PDO
    {
        // Check if connection still alive
        try {
            $this->connection->query('SELECT 1');
        } catch (\Throwable) {
            // Connection lost, reconnect
            $this->connection = $this->createConnection();
        }

        return $this->connection;
    }

    /**
     * Create new PDO connection dengan security options
     */
    private function createConnection(): PDO
    {
        if ($this->dsn === '') {
            throw new RuntimeException('DSN cannot be empty');
        }

        // SQLite allows empty user/pass, MySQL requires user
        $isSqlite = str_starts_with($this->dsn, 'sqlite:');
        if (!$isSqlite && $this->user === '') {
            throw new RuntimeException('DB_USER required for non-SQLite databases');
        }

        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::ATTR_TIMEOUT => 5, // 5 second timeout
        ];

        // Disable multi-statements untuk security (SQL injection mitigation)
        if (defined('PDO::MYSQL_ATTR_MULTI_STATEMENTS')) {
            $options[PDO::MYSQL_ATTR_MULTI_STATEMENTS] = false;
        }

        try {
            return new PDO($this->dsn, $this->user, $this->pass, $options);
        } catch (\PDOException $e) {
            throw new RuntimeException(
                "Failed to connect to database: {$e->getMessage()}",
                (int) $e->getCode(),
                $e
            );
        }
    }

    /**
     * Prevent cloning
     */
    private function __clone()
    {
    }

    /**
     * Prevent unserialization
     */
    public function __wakeup(): void
    {
        throw new RuntimeException('Cannot unserialize singleton');
    }
}
