<?php

declare(strict_types=1);

namespace App;

use PDO;
use RuntimeException;

/**
 * Database connection manager with singleton pattern
 *
 * Reuses single connection per request to avoid connection overhead
 */
final class Db
{
    private static ?PDO $instance = null;

    /**
     * Get PDO instance (singleton)
     *
     * Creates connection on first call, reuses on subsequent calls
     */
    public static function pdo(): PDO
    {
        if (self::$instance !== null) {
            return self::$instance;
        }

        $dsn  = Env::getString('DB_DSN');
        $user = Env::getString('DB_USER');
        $pass = Env::getString('DB_PASS');

        if ($dsn === '') {
            throw new RuntimeException('CONFIG: DB_DSN missing');
        }

        // SQLite allows empty user/pass, MySQL requires user
        $isSqlite = str_starts_with($dsn, 'sqlite:');
        if (!$isSqlite && $user === '') {
            throw new RuntimeException('CONFIG: DB_USER missing');
        }

        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
            PDO::ATTR_PERSISTENT         => false, // Disable persistent for better control
        ];

        if (defined('PDO::MYSQL_ATTR_MULTI_STATEMENTS')) {
            $options[PDO::MYSQL_ATTR_MULTI_STATEMENTS] = false;
        }

        // Set timeout for connection
        if (!$isSqlite && defined('PDO::ATTR_TIMEOUT')) {
            $options[PDO::ATTR_TIMEOUT] = 5;
        }

        self::$instance = new PDO($dsn, $user, $pass, $options);

        return self::$instance;
    }

    /**
     * Close connection (useful for long-running processes)
     */
    public static function close(): void
    {
        self::$instance = null;
    }

    /**
     * Check if connection is alive
     */
    public static function isConnected(): bool
    {
        if (self::$instance === null) {
            return false;
        }

        try {
            self::$instance->query('SELECT 1');
            return true;
        } catch (\Throwable) {
            self::$instance = null;
            return false;
        }
    }
}
