<?php

declare(strict_types=1);

namespace App;

final class Env
{
    /** @var array<string, string> */
    private static array $fileVars = [];

    /** @var array<string, mixed> */
    private static array $securityConfig = [];

    private static bool $booted = false;

    public static function bootstrap(?string $path = null): void
    {
        if (self::$booted) {
            return;
        }
        self::$booted = true;

        $path = $path ?? self::defaultEnvPath();
        if ($path !== null) {
            self::loadFile($path);
        }

        // Load security.json as additional source
        self::loadSecurityJson();
    }

    public static function has(string $key): bool
    {
        if (self::readEnv($key) !== null) {
            return true;
        }
        if (isset(self::$fileVars[$key])) {
            return true;
        }
        return self::hasSecurityConfig($key);
    }

    public static function getString(string $key, string $default = ''): string
    {
        $v = self::readEnv($key);
        if ($v !== null && $v !== '') {
            return $v;
        }

        if (isset(self::$fileVars[$key])) {
            return self::$fileVars[$key];
        }

        // Check security.json
        $secVal = self::getSecurityConfigValue($key);
        if ($secVal !== null && is_string($secVal)) {
            return $secVal;
        }

        return $default;
    }

    public static function getBool(string $key, bool $default = false): bool
    {
        $v = self::getString($key, '');
        if ($v === '') {
            // Check security.json for boolean
            $secVal = self::getSecurityConfigValue($key);
            if ($secVal !== null && is_bool($secVal)) {
                return $secVal;
            }
            return $default;
        }

        $v = strtolower($v);
        return in_array($v, ['1', 'true', 'yes', 'on'], true);
    }

    public static function getInt(string $key, int $default = 0): int
    {
        $v = self::getString($key, '');
        if ($v === '') {
            // Check security.json for int
            $secVal = self::getSecurityConfigValue($key);
            if ($secVal !== null && is_int($secVal)) {
                return $secVal;
            }
            return $default;
        }

        return (int) $v;
    }

    /** @return list<string> */
    public static function getList(string $key): array
    {
        // Check if security.json has array value first
        $secVal = self::getSecurityConfigValue($key);
        if ($secVal !== null && is_array($secVal)) {
            $out = [];
            foreach ($secVal as $item) {
                if (is_string($item) && $item !== '') {
                    $out[] = $item;
                }
            }
            return $out;
        }

        $raw = trim(self::getString($key, ''));
        if ($raw === '') {
            return [];
        }

        $parts = preg_split('/\s*,\s*/', $raw);
        if (!is_array($parts)) {
            return [];
        }

        $out = [];
        foreach ($parts as $p) {
            $p = trim((string) $p);
            if ($p !== '') {
                $out[] = $p;
            }
        }

        return $out;
    }

    private static function readEnv(string $key): ?string
    {
        $v = getenv($key);
        if ($v !== false) {
            return (string) $v;
        }

        if (isset($_ENV[$key]) && is_string($_ENV[$key])) {
            return $_ENV[$key];
        }

        if (isset($_SERVER[$key]) && is_string($_SERVER[$key])) {
            return $_SERVER[$key];
        }

        return null;
    }

    private static function defaultEnvPath(): ?string
    {
        // app/Env.php -> project root = dirname(__DIR__)
        $p = dirname(__DIR__) . '/.env';
        return (is_file($p) && is_readable($p)) ? $p : null;
    }

    private static function loadFile(string $path): void
    {
        $lines = @file($path, FILE_IGNORE_NEW_LINES);
        if (!is_array($lines)) {
            return;
        }

        foreach ($lines as $line) {
            self::parseLine((string) $line);
        }
    }

    private static function parseLine(string $line): void
    {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) {
            return;
        }

        $pos = strpos($line, '=');
        if ($pos === false || $pos < 1) {
            return;
        }

        $key = trim(substr($line, 0, $pos));
        if ($key === '') {
            return;
        }

        $value = trim(substr($line, $pos + 1));
        $value = self::stripInlineComment($value);
        $value = self::unquote($value);

        // jangan override ENV “beneran”
        if (self::readEnv($key) === null) {
            self::$fileVars[$key] = $value;
        }
    }

    private static function stripInlineComment(string $v): string
    {
        $inQuote = '';
        $len = strlen($v);

        for ($i = 0; $i < $len; $i++) {
            $ch = $v[$i];

            if ($inQuote === '') {
                if ($ch === '"' || $ch === "'") {
                    $inQuote = $ch;
                    continue;
                }

                if (($ch === '#' || $ch === ';') && $i > 0) {
                    $prev = $v[$i - 1];
                    if ($prev === ' ' || $prev === "\t") {
                        return rtrim(substr($v, 0, $i));
                    }
                }
            } else {
                if ($ch === $inQuote) {
                    $inQuote = '';
                }
            }
        }

        return $v;
    }

    private static function unquote(string $v): string
    {
        $v = trim($v);
        $len = strlen($v);

        if ($len >= 2) {
            $a = $v[0];
            $b = $v[$len - 1];

            if (($a === '"' && $b === '"') || ($a === "'" && $b === "'")) {
                return substr($v, 1, $len - 2);
            }
        }

        return $v;
    }

    private static function loadSecurityJson(): void
    {
        $securityFile = dirname(__DIR__) . '/data/security.json';
        if (!is_file($securityFile) || !is_readable($securityFile)) {
            return;
        }

        $content = @file_get_contents($securityFile);
        if ($content === false) {
            return;
        }

        $data = json_decode($content, true);
        if (!is_array($data)) {
            return;
        }

        self::$securityConfig = $data;
    }

    /**
     * Map security.json keys to ENV variable names
     */
    private static function getSecurityConfigKey(string $envKey): ?string
    {
        $map = [
            'SHORTLINK_ALLOWED_HOSTS' => 'allowed_hosts',
            'SHORTLINK_DENIED_HOSTS' => 'denied_hosts',
            'SHORTLINK_DENIED_TLDS' => 'denied_tlds',
            'SHORTLINK_HTTPS_ONLY' => 'https_only',
            'SHORTLINK_BLOCK_IP_LITERAL' => 'block_ip_literal',
            'SHORTLINK_BLOCK_USERINFO' => 'block_userinfo',

            'IP_INTEL_ENABLED' => 'ip_intel_enabled',
            'RISK_THRESHOLD' => 'risk_threshold',
            'SHORTLINK_SAFE_URL' => 'safe_url',
            'SHORTLINK_FALLBACK_URL' => 'fallback_url',
            'BLACKBOX_API_URL' => 'blackbox_api_url',
            'IPQUERY_API_URL' => 'ipquery_api_url',
            'IPQUERY_API_KEY' => 'ipquery_api_key',
            'ATLAS_API_URL' => 'atlas_api_url',

            'URL_SIGN_ENABLED' => 'sign_enabled',
            'URL_SIGN_SECRET' => 'sign_secret',
            'URL_SIGN_TTL' => 'sign_ttl',
        ];

        return $map[$envKey] ?? null;
    }

    private static function hasSecurityConfig(string $envKey): bool
    {
        $jsonKey = self::getSecurityConfigKey($envKey);
        if ($jsonKey === null) {
            return false;
        }

        return isset(self::$securityConfig[$jsonKey]);
    }

    private static function getSecurityConfigValue(string $envKey): mixed
    {
        $jsonKey = self::getSecurityConfigKey($envKey);
        if ($jsonKey === null) {
            return null;
        }

        return self::$securityConfig[$jsonKey] ?? null;
    }
}
