<?php

declare(strict_types=1);

namespace App;

/**
 * Facebook Protection Layer
 *
 * Mini WAF to prevent domains from being blocked/banned by Facebook
 *
 * Strategies:
 * 1. User-Agent based response variation
 * 2. Rate limiting per domain for Facebook bots
 * 3. Content validation and safety checks
 * 4. Referrer validation
 * 5. IP-based trust scoring
 * 6. Domain health monitoring
 * 7. Redirect strategy optimization
 */
final class FacebookProtection
{
    private const FACEBOOK_IP_RANGES = [
        // Facebook crawler IP ranges (update periodically)
        '31.13.24.0/21',
        '31.13.64.0/18',
        '66.220.144.0/20',
        '69.63.176.0/20',
        '69.171.224.0/19',
        '74.119.76.0/22',
        '173.252.64.0/18',
        '204.15.20.0/22',
    ];

    private const RATE_LIMIT_WINDOW = 60; // seconds
    private const RATE_LIMIT_MAX_REQUESTS = 30; // per domain per window
    private const SUSPICIOUS_RATE_THRESHOLD = 10; // requests per second

    /**
     * Check if domain is safe to use with Facebook
     *
     * @param string $domain Domain to check
     * @return array{safe: bool, reason: string}
     */
    public static function isDomainSafe(string $domain): array
    {
        // Check domain reputation cache
        $cacheKey = "fb_domain_safe:{$domain}";
        $cached = Cache::fetch($cacheKey, $success);
        if ($success) {
            return $cached;
        }

        $issues = [];

        // Check if domain is in cooldown period
        $cooldownKey = "fb_cooldown:{$domain}";
        if (Cache::exists($cooldownKey)) {
            $issues[] = 'Domain in cooldown after high traffic';
        }

        // Check recent ban reports
        $banKey = "fb_banned:{$domain}";
        if (Cache::exists($banKey)) {
            $issues[] = 'Domain reported as banned by Facebook';
        }

        // Check rate limit violations
        $violationKey = "fb_violation:{$domain}";
        if (Cache::exists($violationKey)) {
            $issues[] = 'Recent rate limit violations detected';
        }

        $result = [
            'safe' => empty($issues),
            'reason' => empty($issues) ? 'Domain is healthy' : implode(', ', $issues),
        ];

        // Cache for 5 minutes
        Cache::store($cacheKey, $result, 300);

        return $result;
    }

    /**
     * Check rate limit for Facebook bot traffic per domain
     *
     * @param string $domain Domain being accessed
     * @param string $userAgent User-Agent header
     * @return bool True if within rate limit, false if exceeded
     */
    public static function checkRateLimit(string $domain, string $userAgent): bool
    {
        if (!Http::isFacebookBot($userAgent)) {
            return true; // No rate limit for non-bot traffic
        }

        $key = "fb_rate:{$domain}";
        $requests = Cache::fetch($key, $success) ?: [];
        $now = time();

        // Clean old requests outside window
        $requests = array_filter($requests, fn($ts) => $ts > ($now - self::RATE_LIMIT_WINDOW));

        // Check if limit exceeded
        if (count($requests) >= self::RATE_LIMIT_MAX_REQUESTS) {
            // Mark violation
            Cache::store("fb_violation:{$domain}", true, 600); // 10 min cooldown
            return false;
        }

        // Check for suspicious burst (too fast)
        $recentRequests = array_filter($requests, fn($ts) => $ts > ($now - 1));
        if (count($recentRequests) >= self::SUSPICIOUS_RATE_THRESHOLD) {
            Cache::store("fb_violation:{$domain}", true, 300); // 5 min cooldown
            return false;
        }

        // Add current request
        $requests[] = $now;
        Cache::store($key, $requests, self::RATE_LIMIT_WINDOW);

        return true;
    }

    /**
     * Get optimal redirect strategy for Facebook bot
     *
     * @param bool $isFacebookBot Is request from Facebook bot
     * @return array{method: string, delay: int, use_meta: bool}
     */
    public static function getRedirectStrategy(bool $isFacebookBot): array
    {
        if ($isFacebookBot) {
            // For bots: Use OG page with meta refresh (no HTTP redirect)
            return [
                'method' => 'og_page',
                'delay' => 0,
                'use_meta' => false, // OG page shows preview, bot doesn't follow redirect
            ];
        }

        // For humans: Fast HTTP redirect
        return [
            'method' => 'http_redirect',
            'delay' => 0,
            'use_meta' => false,
        ];
    }

    /**
     * Validate referrer to detect legitimate Facebook traffic
     *
     * @param string $referrer Referrer header
     * @return bool True if referrer is legitimate Facebook domain
     */
    public static function isLegitimateReferrer(string $referrer): bool
    {
        if ($referrer === '') {
            return true; // Allow empty referrer (direct access)
        }

        $parts = parse_url($referrer);
        if (!is_array($parts)) {
            return false;
        }

        $host = strtolower((string) ($parts['host'] ?? ''));

        $allowedHosts = [
            'facebook.com',
            'fb.com',
            'm.facebook.com',
            'l.facebook.com',
            'lm.facebook.com',
            'instagram.com',
            'l.instagram.com',
            'threads.net',
            'l.threads.net',
            'messenger.com',
            'l.messenger.com',
            'whatsapp.com',
        ];

        foreach ($allowedHosts as $allowed) {
            if ($host === $allowed || str_ends_with($host, '.' . $allowed)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP is from Facebook infrastructure
     *
     * @param string $ip IP address to check
     * @return bool True if IP is in Facebook ranges
     */
    public static function isFacebookIP(string $ip): bool
    {
        // Validate IP format
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) === false) {
            return false;
        }

        $ipLong = ip2long($ip);
        if ($ipLong === false) {
            return false;
        }

        foreach (self::FACEBOOK_IP_RANGES as $range) {
            if (self::ipInRange($ip, $range)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP is in CIDR range
     *
     * @param string $ip IP address
     * @param string $cidr CIDR notation (e.g., 192.168.1.0/24)
     * @return bool True if IP is in range
     */
    private static function ipInRange(string $ip, string $cidr): bool
    {
        [$subnet, $mask] = explode('/', $cidr);

        $ipLong = ip2long($ip);
        $subnetLong = ip2long($subnet);
        $maskLong = -1 << (32 - (int) $mask);

        return ($ipLong & $maskLong) === ($subnetLong & $maskLong);
    }

    /**
     * Validate destination URL is not spammy/malicious
     *
     * @param string $url Destination URL
     * @return array{safe: bool, reason: string}
     */
    public static function validateDestination(string $url): array
    {
        $parts = parse_url($url);
        if (!is_array($parts)) {
            return ['safe' => false, 'reason' => 'Invalid URL format'];
        }

        $host = strtolower((string) ($parts['host'] ?? ''));
        $path = strtolower((string) ($parts['path'] ?? ''));

        // Check for suspicious TLDs commonly used for spam
        $suspiciousTlds = ['.tk', '.ml', '.ga', '.cf', '.gq', '.xyz', '.top'];
        foreach ($suspiciousTlds as $tld) {
            if (str_ends_with($host, $tld)) {
                return ['safe' => false, 'reason' => 'Suspicious TLD: ' . $tld];
            }
        }

        // Check for malicious patterns in path
        $maliciousPatterns = [
            '/phishing',
            '/scam',
            '/fraud',
            '/hack',
            '/crack',
            '/warez',
            '/malware',
        ];

        foreach ($maliciousPatterns as $pattern) {
            if (str_contains($path, $pattern)) {
                return ['safe' => false, 'reason' => 'Suspicious path pattern'];
            }
        }

        // Check if destination is blacklisted
        $blacklistKey = "blacklist:{$host}";
        if (Cache::exists($blacklistKey)) {
            return ['safe' => false, 'reason' => 'Domain is blacklisted'];
        }

        return ['safe' => true, 'reason' => 'Destination appears safe'];
    }

    /**
     * Mark domain as potentially banned (triggered by user report or detection)
     *
     * @param string $domain Domain to mark
     * @param int $ttl Time to live in seconds (default 1 hour)
     */
    public static function markDomainBanned(string $domain, int $ttl = 3600): void
    {
        Cache::store("fb_banned:{$domain}", true, $ttl);

        // Clear safety cache
        Cache::delete("fb_domain_safe:{$domain}");
    }

    /**
     * Put domain in cooldown period (reduce traffic temporarily)
     *
     * @param string $domain Domain to cooldown
     * @param int $ttl Time to live in seconds (default 10 minutes)
     */
    public static function cooldownDomain(string $domain, int $ttl = 600): void
    {
        Cache::store("fb_cooldown:{$domain}", true, $ttl);

        // Clear safety cache
        Cache::delete("fb_domain_safe:{$domain}");
    }

    /**
     * Get domain health metrics
     *
     * @param string $domain Domain to check
     * @return array{requests: int, violations: bool, banned: bool, cooldown: bool}
     */
    public static function getDomainHealth(string $domain): array
    {
        $rateKey = "fb_rate:{$domain}";
        $requests = Cache::fetch($rateKey, $success) ?: [];
        $now = time();
        $recentRequests = array_filter($requests, fn($ts) => $ts > ($now - 60));

        return [
            'requests_per_minute' => count($recentRequests),
            'violations' => Cache::exists("fb_violation:{$domain}"),
            'banned' => Cache::exists("fb_banned:{$domain}"),
            'cooldown' => Cache::exists("fb_cooldown:{$domain}"),
        ];
    }

    /**
     * Generate varied response to avoid fingerprinting
     *
     * @param string $userAgent User-Agent header
     * @return array{cache_control: string, vary_header: string}
     */
    public static function getResponseHeaders(string $userAgent): array
    {
        if (Http::isFacebookBot($userAgent)) {
            // For bots: Allow caching but with variation
            return [
                'cache_control' => 'public, max-age=300, must-revalidate',
                'vary_header' => 'User-Agent, Accept-Language',
            ];
        }

        // For humans: No cache
        return [
            'cache_control' => 'no-store, no-cache, must-revalidate, private',
            'vary_header' => 'User-Agent',
        ];
    }

    /**
     * Check if request passes all protection checks
     *
     * @param string $domain Domain being accessed
     * @param string $userAgent User-Agent header
     * @param string $ip Client IP
     * @param string $referrer Referrer header
     * @return array{allowed: bool, reason: string, action: string}
     */
    public static function validateRequest(
        string $domain,
        string $userAgent,
        string $ip,
        string $referrer
    ): array {
        $isFbBot = Http::isFacebookBot($userAgent);

        // Check 1: Domain health
        $domainCheck = self::isDomainSafe($domain);
        if (!$domainCheck['safe'] && $isFbBot) {
            return [
                'allowed' => false,
                'reason' => 'Domain health: ' . $domainCheck['reason'],
                'action' => 'use_fallback_domain',
            ];
        }

        // Check 2: Rate limiting (for Facebook bots)
        if ($isFbBot && !self::checkRateLimit($domain, $userAgent)) {
            return [
                'allowed' => false,
                'reason' => 'Rate limit exceeded for domain',
                'action' => 'cooldown_and_use_fallback',
            ];
        }

        // Check 3: IP validation (for Facebook bots)
        if ($isFbBot && !self::isFacebookIP($ip)) {
            // Not from Facebook IP but claims to be Facebook bot - suspicious
            return [
                'allowed' => true, // Allow but log
                'reason' => 'Suspicious: FB bot UA but non-FB IP',
                'action' => 'log_suspicious',
            ];
        }

        // All checks passed
        return [
            'allowed' => true,
            'reason' => 'All protection checks passed',
            'action' => 'proceed',
        ];
    }
}
