<?php

declare(strict_types=1);

namespace App;

use App\Domain\Domain\DomainRepositoryInterface;
use App\Infrastructure\Persistence\AbstractDomainRepository;
use PDO;

/**
 * Repository for global_domains table
 *
 * Admin-managed domains accessible to ALL users
 */
final class GlobalDomainRepository extends AbstractDomainRepository implements DomainRepositoryInterface
{
    protected string $tableName = 'global_domains';

    /**
     * @var list<string>
     */
    protected array $allowedUpdateFields = [
        'domain',
        'provider',
        'is_wildcard',
        'is_active',
        'is_default',
        'cpanel_host',
        'cpanel_username',
        'cpanel_token',
        'cf_zone_id',
        'cf_api_token',
        'cf_proxied',
        'dns_record_id',
        'target_ip',
        'target_cname',
        'last_sync_at',
    ];

    public function __construct(PDO $pdo)
    {
        parent::__construct($pdo);
    }

    /**
     * Create new global domain
     *
     * @param array{
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   is_default?: int,
     *   cpanel_host?: string,
     *   cpanel_username?: string,
     *   cpanel_token?: string,
     *   cf_zone_id?: string,
     *   cf_api_token?: string,
     *   cf_proxied?: int,
     *   target_ip?: string,
     *   target_cname?: string
     * } $data
     * @return int Inserted ID
     */
    public function create(array $data): int
    {
        // If this is set as default, unset all other defaults first
        if (($data['is_default'] ?? 0) === 1) {
            $this->clearDefaultFlag();
        }

        $sql = 'INSERT INTO global_domains
                (domain, provider, is_wildcard, is_default, cpanel_host, cpanel_username, cpanel_token,
                 cf_zone_id, cf_api_token, cf_proxied, target_ip, target_cname)
                VALUES
                (:domain, :provider, :is_wildcard, :is_default, :cpanel_host, :cpanel_username, :cpanel_token,
                 :cf_zone_id, :cf_api_token, :cf_proxied, :target_ip, :target_cname)';

        $params = $this->getCommonInsertParams($data);
        $params['is_default'] = $data['is_default'] ?? 0;

        return $this->executeInsert($sql, $params);
    }

    /**
     * List all global domains
     *
     * @param bool|null $isActive Filter by active status (null = all)
     * @return list<array{
     *   id: int,
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   is_active: int,
     *   is_default: int,
     *   target_ip: string|null,
     *   target_cname: string|null,
     *   created_at: string,
     *   updated_at: string
     * }>
     */
    public function listAll(?bool $isActive = null): array
    {
        $sql = 'SELECT id, domain, provider, is_wildcard, is_active, is_default,
                       target_ip, target_cname, created_at, updated_at
                FROM global_domains';

        $params = [];

        if ($isActive !== null) {
            $sql .= ' WHERE is_active = :is_active';
            $params['is_active'] = $isActive ? 1 : 0;
        }

        $sql .= ' ORDER BY is_default DESC, created_at DESC';

        /** @var list<array{id: int, domain: string, provider: string, is_wildcard: int, is_active: int, is_default: int, target_ip: string|null, target_cname: string|null, created_at: string, updated_at: string}> */
        return $this->fetchAll($sql, $params);
    }

    /**
     * List active global domains (for dropdown)
     *
     * @return list<array{id: int, domain: string, is_wildcard: int, is_default: int}>
     */
    public function listActive(): array
    {
        $sql = 'SELECT id, domain, is_wildcard, is_default
                FROM global_domains
                WHERE is_active = 1
                ORDER BY is_default DESC, domain ASC';

        /** @var list<array{id: int, domain: string, is_wildcard: int, is_default: int}> */
        return $this->fetchAll($sql);
    }

    /**
     * Get domain by ID
     *
     * @param int $id
     * @return array{
     *   id: int,
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   is_active: int,
     *   is_default: int,
     *   cpanel_host: string|null,
     *   cpanel_username: string|null,
     *   cpanel_token: string|null,
     *   cf_zone_id: string|null,
     *   cf_api_token: string|null,
     *   cf_proxied: int,
     *   dns_record_id: string|null,
     *   target_ip: string|null,
     *   target_cname: string|null,
     *   last_sync_at: string|null,
     *   created_at: string,
     *   updated_at: string
     * }|null
     */
    public function findById(int $id): ?array
    {
        $sql = 'SELECT * FROM global_domains WHERE id = :id LIMIT 1';

        /** @var array{id: int, domain: string, provider: string, is_wildcard: int, is_active: int, is_default: int, cpanel_host: string|null, cpanel_username: string|null, cpanel_token: string|null, cf_zone_id: string|null, cf_api_token: string|null, cf_proxied: int, dns_record_id: string|null, target_ip: string|null, target_cname: string|null, last_sync_at: string|null, created_at: string, updated_at: string}|null */
        return $this->fetchOne($sql, ['id' => $id]);
    }

    /**
     * Get default domain
     *
     * @return array{id: int, domain: string}|null
     */
    public function getDefault(): ?array
    {
        $sql = 'SELECT id, domain FROM global_domains
                WHERE is_default = 1 AND is_active = 1
                LIMIT 1';

        /** @var array{id: int, domain: string}|null */
        return $this->fetchOne($sql, []);
    }

    /**
     * Update domain
     *
     * @param int $id
     * @param array<string, mixed> $data Fields to update
     * @return bool Success
     */
    public function update(int $id, array $data): bool
    {
        // If setting as default, unset all other defaults first
        if (isset($data['is_default']) && (int) $data['is_default'] === 1) {
            $this->clearDefaultFlag();
        }

        $updateData = $this->buildUpdateQuery($data, ['id' => $id]);
        if ($updateData === null) {
            return false;
        }

        $sql = 'UPDATE global_domains SET ' . $updateData['sql'] . '
                WHERE id = :id LIMIT 1';

        return $this->executeUpdate($sql, $updateData['params']);
    }

    /**
     * Delete domain by ID (implements interface)
     *
     * @param int $id
     * @return bool Success
     */
    public function deleteById(int $id): bool
    {
        return $this->delete($id);
    }

    /**
     * Delete domain
     *
     * @param int $id
     * @return bool Success
     */
    public function delete(int $id): bool
    {
        $sql = 'DELETE FROM global_domains WHERE id = :id LIMIT 1';
        return $this->executeDelete($sql, ['id' => $id]);
    }

    /**
     * Check if domain exists by name (implements interface)
     *
     * @param string $domain
     * @return bool
     */
    public function domainExists(string $domain): bool
    {
        return $this->exists($domain);
    }

    /**
     * Check if domain exists
     *
     * @param string $domain
     * @return bool
     */
    public function exists(string $domain): bool
    {
        $sql = 'SELECT 1 FROM global_domains WHERE domain = :domain LIMIT 1';
        return $this->rowExists($sql, ['domain' => $domain]);
    }

    /**
     * Clear default flag from all domains
     */
    private function clearDefaultFlag(): void
    {
        $stmt = $this->pdo->prepare('UPDATE global_domains SET is_default = 0 WHERE is_default = 1');
        $stmt->execute();
    }
}
