<?php

declare(strict_types=1);

namespace App;

final class Http
{
    /**
     * Detect Facebook/Meta family bots (Facebook, Instagram, WhatsApp, Threads, Messenger)
     *
     * @param string $userAgent User-Agent header
     * @return bool True if bot detected
     */
    public static function isFacebookBot(string $userAgent): bool
    {
        $ua = strtolower($userAgent);

        if ($ua === '') {
            return false;
        }

        // Facebook family bot user agents
        $needles = [
            // Facebook
            'facebookexternalhit',
            'facebot',
            'meta-externalagent',
            'meta-externalfetcher',

            // Instagram
            'instagrambot',
            'instagram',

            // WhatsApp
            'whatsapp',

            // Threads
            'threadsbot',
            'threads',

            // Messenger
            'messengerbot',
            'fbmessenger',
        ];

        foreach ($needles as $n) {
            if (strpos($ua, $n) !== false) {
                return true;
            }
        }

        return false;
    }

    public static function e(string $s): string
    {
        return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
    }

    public static function nonce(): string
    {
        return rtrim(strtr(base64_encode(random_bytes(16)), '+/', '-_'), '=');
    }

    public static function sendSecurityHeaders(string $nonce, bool $isOgPage): void
    {
        header('X-Content-Type-Options: nosniff');
        header('Referrer-Policy: no-referrer');
        header('X-Frame-Options: DENY');
        header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
        header('Cross-Origin-Opener-Policy: same-origin');
        header('Cross-Origin-Resource-Policy: same-site');

        if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') {
            header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
        }

        if ($isOgPage) {
            header('X-Robots-Tag: noindex, nofollow');
            header('Vary: User-Agent');

            $csp = "default-src 'none'; "
                . "img-src https:; "
                . "style-src 'nonce-" . $nonce . "'; "
                . "base-uri 'none'; "
                . "form-action 'none'; "
                . "frame-ancestors 'none'";
            header('Content-Security-Policy: ' . $csp);
        }
    }

    public static function validateCode(string $code): string
    {
        $code = trim($code);

        if (!preg_match('/^[A-Za-z0-9_-]{3,32}$/', $code)) {
            return '';
        }

        return $code;
    }

    public static function validateUrl(string $url, bool $allowHttp): string
    {
        $url = trim($url);

        if ($url === '') {
            return '';
        }

        if (preg_match('/[\r\n]/', $url)) {
            return '';
        }

        if (filter_var($url, FILTER_VALIDATE_URL) === false) {
            return '';
        }

        $parts = parse_url($url);
        if (!is_array($parts)) {
            return '';
        }

        $scheme = strtolower((string) ($parts['scheme'] ?? ''));
        if ($allowHttp) {
            if (!in_array($scheme, ['http', 'https'], true)) {
                return '';
            }
        } else {
            if ($scheme !== 'https') {
                return '';
            }
        }

        return $url;
    }

    /**
     * @param list<string> $allowedHosts
     */
    public static function isAllowedHost(string $url, array $allowedHosts): bool
    {
        if ($allowedHosts === []) {
            return true;
        }

        $parts = parse_url($url);
        if (!is_array($parts)) {
            return false;
        }

        $host = strtolower((string) ($parts['host'] ?? ''));
        if ($host === '') {
            return false;
        }

        foreach ($allowedHosts as $h) {
            if ($host === strtolower($h)) {
                return true;
            }
        }

        return false;
    }
}
