<?php

declare(strict_types=1);

namespace App\IpIntel;

use Throwable;

final class AtlasClient
{
    private const DEFAULT_BASE_URL = 'https://ip2country.ipinfo.app';

    public function __construct(
        private readonly HttpClientInterface $http,
        private readonly string $baseUrl = self::DEFAULT_BASE_URL,
        private readonly int $timeoutSeconds = 3
    ) {
    }

    /**
     * @return array{
     *   success: bool,
     *   ip: string,
     *   country_code: string|null,
     *   country: string|null,
     *   continent: string|null,
     *   asn: int|null,
     *   asn_org: string|null,
     *   error: string|null
     * }
     */
    public function lookup(string $ip): array
    {
        $ip = IpValidator::assertValidIp($ip);

        $url = rtrim($this->baseUrl, '/') . '/api/v2/ip/' . rawurlencode($ip);

        $resp = $this->http->get($url, [
            'Accept' => 'application/json',
        ], $this->timeoutSeconds);

        if ($resp->statusCode !== 200) {
            return $this->fail($ip, 'HTTP_' . $resp->statusCode);
        }

        try {
            $data = json_decode($resp->body, true, 512, JSON_THROW_ON_ERROR);
        } catch (Throwable $e) {
            return $this->fail($ip, 'JSON_PARSE_FAILED');
        }

        if (!is_array($data)) {
            return $this->fail($ip, 'JSON_SHAPE_INVALID');
        }

        $error = $this->asNullableString($data['error'] ?? null);
        if ($error !== null && $error !== '') {
            return $this->fail($ip, $error);
        }

        return [
            'success' => true,
            'ip' => $ip,
            'country_code' => $this->asNullableString($data['country_code'] ?? null),
            'country' => $this->asNullableString($data['country'] ?? null),
            'continent' => $this->asNullableString($data['continent'] ?? null),
            'asn' => $this->asNullableInt($data['as_number'] ?? null),
            'asn_org' => $this->asNullableString($data['as_description'] ?? null),
            'error' => null,
        ];
    }

    /**
     * @return array{
     *   success: bool,
     *   ip: string,
     *   country_code: string|null,
     *   country: string|null,
     *   continent: string|null,
     *   asn: int|null,
     *   asn_org: string|null,
     *   error: string|null
     * }
     */
    private function fail(string $ip, string $reason): array
    {
        return [
            'success' => false,
            'ip' => $ip,
            'country_code' => null,
            'country' => null,
            'continent' => null,
            'asn' => null,
            'asn_org' => null,
            'error' => $reason,
        ];
    }

    private function asNullableString(mixed $v): ?string
    {
        if (!is_string($v)) {
            return null;
        }

        $v = trim($v);
        return $v === '' ? null : $v;
    }

    private function asNullableInt(mixed $v): ?int
    {
        if (is_int($v)) {
            return $v;
        }
        if (is_string($v) && ctype_digit($v)) {
            return (int) $v;
        }

        return null;
    }
}
