<?php

declare(strict_types=1);

namespace App\IpIntel;

use RuntimeException;
use Throwable;

final class CurlHttpClient implements HttpClientInterface
{
    private const MAX_BODY_BYTES = 1024 * 1024; // 1MB hard cap

    public function get(string $url, array $headers, int $timeoutSeconds): HttpResponse
    {
        $this->assertHttpsUrl($url);

        $ch = curl_init($url);
        if ($ch === false) {
            throw new RuntimeException('Failed to init cURL');
        }

        $headerLines = $this->normalizeHeaders($headers);
        $timeoutSeconds = max(1, min(30, $timeoutSeconds));

        try {
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => false,
                CURLOPT_MAXREDIRS => 0,
                CURLOPT_CONNECTTIMEOUT => $timeoutSeconds,
                CURLOPT_TIMEOUT => $timeoutSeconds,
                CURLOPT_PROTOCOLS => CURLPROTO_HTTPS,
                CURLOPT_REDIR_PROTOCOLS => CURLPROTO_HTTPS,
                CURLOPT_HTTPHEADER => $headerLines,
                CURLOPT_USERAGENT => 'OGShortlink/1.0',
                CURLOPT_ENCODING => '',
                CURLOPT_SSL_VERIFYPEER => true,
                CURLOPT_SSL_VERIFYHOST => 2,
            ]);

            $body = curl_exec($ch);
            if (!is_string($body)) {
                return new HttpResponse(0, '');
            }

            if (strlen($body) > self::MAX_BODY_BYTES) {
                return new HttpResponse(0, '');
            }

            $status = (int) curl_getinfo($ch, CURLINFO_RESPONSE_CODE);

            return new HttpResponse($status, $body);
        } catch (Throwable $e) {
            return new HttpResponse(0, '');
        } finally {
            curl_close($ch);
        }
    }

    private function assertHttpsUrl(string $url): void
    {
        $parts = parse_url($url);
        if (!is_array($parts)) {
            throw new RuntimeException('Invalid URL');
        }

        $scheme = strtolower((string) ($parts['scheme'] ?? ''));
        if ($scheme !== 'https') {
            throw new RuntimeException('Only https:// URLs are allowed');
        }

        $host = (string) ($parts['host'] ?? '');
        if ($host === '') {
            throw new RuntimeException('URL host is required');
        }
    }

    /**
     * @param array<string, string> $headers
     * @return list<string>
     */
    private function normalizeHeaders(array $headers): array
    {
        $out = [];
        foreach ($headers as $k => $v) {
            $k = trim((string) $k);
            $v = trim((string) $v);
            if ($k === '' || $v === '') {
                continue;
            }
            $out[] = $k . ': ' . $v;
        }

        return $out;
    }
}
