<?php

declare(strict_types=1);

namespace App\IpIntel;

use Throwable;

final class Ip2AsnClient
{
    private const DEFAULT_BASE_URL = 'https://ip2asn.ipinfo.app';

    public function __construct(
        private readonly HttpClientInterface $http,
        private readonly string $baseUrl = self::DEFAULT_BASE_URL,
        private readonly int $timeoutSeconds = 3
    ) {
    }

    /**
     * @return array{
     *   success: bool,
     *   ip: string,
     *   asn: int|null,
     *   asn_name: string|null,
     *   subnet: string|null,
     *   error: string|null
     * }
     */
    public function lookup(string $ip): array
    {
        $ip = IpValidator::assertValidIp($ip);

        $url = rtrim($this->baseUrl, '/') . '/lookup/' . rawurlencode($ip);

        $resp = $this->http->get($url, [
            'Accept' => 'application/json',
        ], $this->timeoutSeconds);

        if ($resp->statusCode !== 200) {
            return $this->fail($ip, 'HTTP_' . $resp->statusCode);
        }

        try {
            $data = json_decode($resp->body, true, 512, JSON_THROW_ON_ERROR);
        } catch (Throwable $e) {
            return $this->fail($ip, 'JSON_PARSE_FAILED');
        }

        if (!is_array($data)) {
            return $this->fail($ip, 'JSON_SHAPE_INVALID');
        }

        $error = $this->asNullableString($data['error'] ?? null);
        if ($error !== null && $error !== '') {
            return $this->fail($ip, $error);
        }

        $announcedBy = $data['announcedBy'] ?? null;
        if (!is_array($announcedBy) || $announcedBy === []) {
            return $this->fail($ip, 'NO_ANNOUNCER');
        }

        $first = $announcedBy[0] ?? null;
        if (!is_array($first)) {
            return $this->fail($ip, 'ANNOUNCER_SHAPE_INVALID');
        }

        $asn = $this->asNullableInt($first['asn'] ?? null);
        $name = $this->asNullableString($first['name'] ?? null);
        $subnet = $this->asNullableString($first['subnet'] ?? null);

        return [
            'success' => true,
            'ip' => $ip,
            'asn' => $asn,
            'asn_name' => $name,
            'subnet' => $subnet,
            'error' => null,
        ];
    }

    /**
     * @return array{
     *   success: bool,
     *   ip: string,
     *   asn: int|null,
     *   asn_name: string|null,
     *   subnet: string|null,
     *   error: string|null
     * }
     */
    private function fail(string $ip, string $reason): array
    {
        return [
            'success' => false,
            'ip' => $ip,
            'asn' => null,
            'asn_name' => null,
            'subnet' => null,
            'error' => $reason,
        ];
    }

    private function asNullableString(mixed $v): ?string
    {
        if (!is_string($v)) {
            return null;
        }

        $v = trim($v);
        return $v === '' ? null : $v;
    }

    private function asNullableInt(mixed $v): ?int
    {
        if (is_int($v)) {
            return $v;
        }
        if (is_string($v) && ctype_digit($v)) {
            return (int) $v;
        }

        return null;
    }
}
