<?php

declare(strict_types=1);

namespace App\IpIntel;

use Throwable;

final class IpQueryClient
{
    public function __construct(
        private readonly HttpClientInterface $http,
        private readonly string $apiBase = 'https://api.ipquery.io',
        private readonly string $apiKey = '',
        private readonly int $timeout = 3,
    ) {
    }

    /**
     * @return array{
     *   success: bool,
     *   flags: array<string, bool>,
     *   base_risk_score: int,
     *   country: ?string,
     *   country_code: ?string,
     *   asn: ?int,
     *   asn_org: ?string
     * }
     */
    public function query(string $ip): array
    {
        $url = rtrim($this->apiBase, '/') . '/' . rawurlencode($ip);

        $headers = ['Accept' => 'application/json'];
        if ($this->apiKey !== '') {
            // Some deployments gate by key; harmless if ignored.
            $headers['X-API-Key'] = $this->apiKey;
        }

        try {
            $resp = $this->http->get($url, $headers, $this->timeout);

            if ($resp->statusCode !== 200) {
                return [
                    'success' => false,
                    'flags' => [],
                    'base_risk_score' => 0,
                    'country' => null,
                    'country_code' => null,
                    'asn' => null,
                    'asn_org' => null,
                ];
            }

            $json = json_decode($resp->body, true, 512, JSON_THROW_ON_ERROR);
            if (!is_array($json)) {
                return [
                    'success' => false,
                    'flags' => [],
                    'base_risk_score' => 0,
                    'country' => null,
                    'country_code' => null,
                    'asn' => null,
                    'asn_org' => null,
                ];
            }

            $risk = $json['risk'] ?? [];
            $isp = $json['isp'] ?? [];
            $location = $json['location'] ?? [];

            if (!is_array($risk)) {
                $risk = [];
            }
            if (!is_array($isp)) {
                $isp = [];
            }
            if (!is_array($location)) {
                $location = [];
            }

            $flags = [
                'is_tor' => (bool) (($risk['is_tor'] ?? false) === true),
                'is_proxy' => (bool) (($risk['is_proxy'] ?? false) === true),
                'is_vpn' => (bool) (($risk['is_vpn'] ?? false) === true),
                'is_datacenter' => (bool) (($risk['is_datacenter'] ?? false) === true),
            ];

            $baseRisk = (int) ($risk['risk_score'] ?? 0);
            if ($baseRisk < 0) {
                $baseRisk = 0;
            }
            if ($baseRisk > 100) {
                $baseRisk = 100;
            }

            $country = null;
            $countryCode = null;
            if (is_array($location)) {
                $country = is_string($location['country'] ?? null) ? (string) $location['country'] : null;
                $countryCode = is_string($location['country_code'] ?? null) ? (string) $location['country_code'] : null;
            }

            $asn = null;
            $asnOrg = null;
            if (is_array($isp)) {
                $asnRaw = $isp['asn'] ?? null;
                if (is_string($asnRaw)) {
                    $asnRaw = strtoupper($asnRaw);
                    if (preg_match('/^AS(\d+)$/', $asnRaw, $m) === 1) {
                        $asn = (int) $m[1];
                    }
                }

                $org = $isp['org'] ?? null;
                if (is_string($org) && trim($org) !== '') {
                    $asnOrg = trim($org);
                }
            }

            return [
                'success' => true,
                'flags' => $flags,
                'base_risk_score' => $baseRisk,
                'country' => $country,
                'country_code' => $countryCode,
                'asn' => $asn,
                'asn_org' => $asnOrg,
            ];
        } catch (Throwable $e) {
            return [
                'success' => false,
                'flags' => [],
                'base_risk_score' => 0,
                'country' => null,
                'country_code' => null,
                'asn' => null,
                'asn_org' => null,
            ];
        }
    }
}
