<?php

declare(strict_types=1);

namespace App\IpIntel;

final class SfsBlocklistDownloader
{
    private const DEFAULT_BASE_URL = 'https://sfs.ipinfo.app';

    public function __construct(
        private readonly HttpClientInterface $http,
        private readonly string $baseUrl = self::DEFAULT_BASE_URL,
        private readonly int $timeoutSeconds = 5
    ) {
    }

    /**
     * Download "Toxic" htaccess list and extract CIDRs.
     *
     * @return array{success: bool, cidrs: list<string>, error: string|null}
     */
    public function downloadToxicCidrsHtaccess(): array
    {
        $url = rtrim($this->baseUrl, '/') . '/api/downloads/htaccess/toxic/';

        $resp = $this->http->get($url, [
            'Accept' => 'text/plain',
        ], $this->timeoutSeconds);

        if ($resp->statusCode !== 200) {
            return ['success' => false, 'cidrs' => [], 'error' => 'HTTP_' . $resp->statusCode];
        }

        $cidrs = $this->extractCidrs($resp->body);

        return ['success' => true, 'cidrs' => $cidrs, 'error' => null];
    }

    /**
     * @return list<string>
     */
    private function extractCidrs(string $body): array
    {
        $body = str_replace(["\r\n", "\r", "\n"], ' ', $body);

        $matches = [];
        preg_match_all('/\bDeny\s+from\s+([0-9a-fA-F\.:]+\/\d{1,3})\b/', $body, $matches);

        $seen = [];
        $out = [];

        foreach (($matches[1] ?? []) as $cidr) {
            $cidr = trim((string) $cidr);
            if ($cidr === '' || isset($seen[$cidr])) {
                continue;
            }
            if (!Cidr::isValid($cidr)) {
                continue;
            }

            $seen[$cidr] = true;
            $out[] = $cidr;
        }

        return $out;
    }
}
