<?php

declare(strict_types=1);

namespace App;

/**
 * Safe IP resolution - anti header spoofing
 */
final class IpResolver
{
    /**
     * Trusted proxy IPs (Cloudflare, trusted load balancers)
     * @var array<string>
     */
    private static array $trustedProxies = [
        // Cloudflare IPv4 ranges (example - add full list from https://www.cloudflare.com/ips-v4)
        '173.245.48.0/20',
        '103.21.244.0/22',
        '103.22.200.0/22',
        '103.31.4.0/22',
        '141.101.64.0/18',
        '108.162.192.0/18',
        '190.93.240.0/20',
        '188.114.96.0/20',
        '197.234.240.0/22',
        '198.41.128.0/17',
        '162.158.0.0/15',
        '104.16.0.0/13',
        '104.24.0.0/14',
        '172.64.0.0/13',
        '131.0.72.0/22',
    ];

    /**
     * Resolve client IP safely
     */
    public static function resolve(): string
    {
        $remoteAddr = $_SERVER['REMOTE_ADDR'] ?? '';

        // Validate REMOTE_ADDR first
        if (!self::isValidIp($remoteAddr)) {
            return '0.0.0.0'; // Invalid IP - return safe default
        }

        // If REMOTE_ADDR is NOT a trusted proxy, use it directly
        if (!self::isTrustedProxy($remoteAddr)) {
            return $remoteAddr;
        }

        // REMOTE_ADDR is trusted proxy - check forwarded headers
        // Priority: CF-Connecting-IP > X-Forwarded-For (first) > REMOTE_ADDR

        // Cloudflare specific header
        $cfIp = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? '';
        if ($cfIp !== '' && self::isValidIp($cfIp) && !self::isSuspicious($cfIp)) {
            return $cfIp;
        }

        // X-Forwarded-For (take first IP only)
        $xffHeader = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '';
        if ($xffHeader !== '' && !self::isSuspicious($xffHeader)) {
            $ips = array_map('trim', explode(',', $xffHeader));
            $firstIp = $ips[0] ?? '';
            if (self::isValidIp($firstIp)) {
                return $firstIp;
            }
        }

        // Fallback to REMOTE_ADDR (trusted proxy IP)
        return $remoteAddr;
    }

    /**
     * Validate IP format (IPv4 or IPv6)
     */
    private static function isValidIp(string $ip): bool
    {
        if ($ip === '') {
            return false;
        }

        // Reject obviously malformed
        if (strlen($ip) > 45) { // Max IPv6 length
            return false;
        }

        return filter_var($ip, FILTER_VALIDATE_IP) !== false;
    }

    /**
     * Check if IP header contains suspicious patterns
     * Reject: %, [], excessive length, CRLF
     */
    private static function isSuspicious(string $value): bool
    {
        // Length check
        if (strlen($value) > 200) {
            return true;
        }

        // Suspicious characters
        if (
            strpos($value, '%') !== false ||
            strpos($value, '[') !== false ||
            strpos($value, ']') !== false ||
            strpos($value, "\r") !== false ||
            strpos($value, "\n") !== false
        ) {
            return true;
        }

        return false;
    }

    /**
     * Check if IP is in trusted proxy list
     */
    private static function isTrustedProxy(string $ip): bool
    {
        foreach (self::$trustedProxies as $cidr) {
            if (self::ipInRange($ip, $cidr)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP is in CIDR range
     */
    private static function ipInRange(string $ip, string $cidr): bool
    {
        if (strpos($cidr, '/') === false) {
            return $ip === $cidr;
        }

        [$subnet, $mask] = explode('/', $cidr);

        $ipLong = ip2long($ip);
        $subnetLong = ip2long($subnet);

        if ($ipLong === false || $subnetLong === false) {
            return false;
        }

        $maskLong = -1 << (32 - (int) $mask);
        return ($ipLong & $maskLong) === ($subnetLong & $maskLong);
    }

    /**
     * Get IP hash for privacy (SHA-256)
     */
    public static function hash(string $ip): string
    {
        return hash('sha256', $ip);
    }
}
