<?php

declare(strict_types=1);

namespace App\Monitoring;

/**
 * Health check untuk monitoring system health
 */
final class HealthCheck
{
    /**
     * @var array<string, callable(): array{status: string, details?: mixed}>
     */
    private array $checks = [];

    public function addCheck(string $name, callable $check): void
    {
        $this->checks[$name] = $check;
    }

    /**
     * Run all health checks
     *
     * @return array{status: string, timestamp: int, checks: array<string, array{status: string, details?: mixed}>}
     */
    public function run(): array
    {
        $results = [];
        $allHealthy = true;

        foreach ($this->checks as $name => $check) {
            try {
                $result = $check();
                $results[$name] = $result;

                if ($result['status'] !== 'healthy') {
                    $allHealthy = false;
                }
            } catch (\Throwable $e) {
                $results[$name] = [
                    'status' => 'unhealthy',
                    'error' => $e->getMessage(),
                ];
                $allHealthy = false;
            }
        }

        return [
            'status' => $allHealthy ? 'healthy' : 'unhealthy',
            'timestamp' => time(),
            'checks' => $results,
        ];
    }

    /**
     * Database health check
     *
     * @return array{status: string, details?: array{response_time_ms: float}, error?: string}
     */
    public static function checkDatabase(\PDO $pdo): array
    {
        try {
            $start = microtime(true);
            $pdo->query('SELECT 1');
            $duration = microtime(true) - $start;

            return [
                'status' => 'healthy',
                'details' => [
                    'response_time_ms' => round($duration * 1000, 2),
                ],
            ];
        } catch (\Throwable $e) {
            return [
                'status' => 'unhealthy',
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Redis health check
     *
     * @return array{status: string, details?: array{response_time_ms: float, used_memory: string, connected_clients: int}, error?: string}
     */
    public static function checkRedis(\Redis $redis): array
    {
        try {
            $start = microtime(true);
            $redis->ping();
            $duration = microtime(true) - $start;

            $info = $redis->info();

            return [
                'status' => 'healthy',
                'details' => [
                    'response_time_ms' => round($duration * 1000, 2),
                    'used_memory' => $info['used_memory_human'] ?? 'unknown',
                    'connected_clients' => $info['connected_clients'] ?? 0,
                ],
            ];
        } catch (\Throwable $e) {
            return [
                'status' => 'unhealthy',
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Disk space health check
     *
     * @return array{status: string, details?: array{used_percent: float, free_gb: float, total_gb: float}, error?: string}
     */
    public static function checkDiskSpace(string $path = '/', int $threshold = 80): array
    {
        $total = disk_total_space($path);
        $free = disk_free_space($path);

        if ($total === false || $free === false) {
            return [
                'status' => 'unhealthy',
                'error' => 'Cannot read disk space',
            ];
        }

        $used = $total - $free;
        $usedPercent = ($used / $total) * 100;

        $status = $usedPercent < $threshold ? 'healthy' : 'unhealthy';

        return [
            'status' => $status,
            'details' => [
                'used_percent' => round($usedPercent, 2),
                'free_gb' => round($free / 1024 / 1024 / 1024, 2),
                'total_gb' => round($total / 1024 / 1024 / 1024, 2),
            ],
        ];
    }

    /**
     * Memory usage health check
     *
     * @return array{status: string, details: array{used_percent: float, used_mb: float, limit: string}}
     */
    public static function checkMemory(int $threshold = 80): array
    {
        $limit = ini_get('memory_limit');
        $limitBytes = self::parseMemoryLimit($limit);

        $used = memory_get_usage(true);
        $usedPercent = ($used / $limitBytes) * 100;

        $status = $usedPercent < $threshold ? 'healthy' : 'unhealthy';

        return [
            'status' => $status,
            'details' => [
                'used_percent' => round($usedPercent, 2),
                'used_mb' => round($used / 1024 / 1024, 2),
                'limit' => $limit,
            ],
        ];
    }

    private static function parseMemoryLimit(string $limit): int
    {
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit) - 1]);
        $value = (int) $limit;

        return match ($last) {
            'g' => $value * 1024 * 1024 * 1024,
            'm' => $value * 1024 * 1024,
            'k' => $value * 1024,
            default => $value,
        };
    }
}
