<?php

declare(strict_types=1);

namespace App;

/**
 * Rate limiter using sliding window counter algorithm
 *
 * Uses two fixed windows with interpolation to approximate sliding window.
 * Memory-efficient (2 counters vs N timestamps) and race-condition safe
 * with atomic increments.
 */
final class RateLimiter
{
    /**
     * Check if request is allowed based on rate limit
     *
     * Uses sliding window counter: combines current and previous window
     * counts weighted by how far into current window we are.
     *
     * @param string $id Unique identifier (e.g., IP address, user ID)
     * @param int $max Maximum requests allowed in window
     * @param int $window Time window in seconds
     * @return bool True if request is allowed, false if rate limit exceeded
     */
    public function check(string $id, int $max = 100, int $window = 60): bool
    {
        $now = time();

        // Calculate current and previous window boundaries
        $currentWindow = (int) floor($now / $window);
        $previousWindow = $currentWindow - 1;

        // How far into current window (0.0 to 1.0)
        $windowProgress = ($now % $window) / $window;

        // Keys for current and previous windows
        $currentKey = "rl:{$id}:{$currentWindow}";
        $previousKey = "rl:{$id}:{$previousWindow}";

        // Get counts (previous window may not exist)
        $previousCount = (int) (Cache::get($previousKey) ?? 0);
        $currentCount = (int) (Cache::get($currentKey) ?? 0);

        // Calculate weighted count (sliding window approximation)
        // Previous window's contribution decreases as we progress through current window
        $weightedCount = (int) floor($previousCount * (1 - $windowProgress)) + $currentCount;

        // Check if adding this request would exceed limit
        if ($weightedCount >= $max) {
            return false;
        }

        // Increment current window counter atomically
        // TTL = 2 * window to ensure previous window data available
        Cache::increment($currentKey, 1, $window * 2);

        return true;
    }

    /**
     * Get current request count for an identifier (weighted)
     *
     * @param string $id Unique identifier
     * @param int $window Time window in seconds
     * @return int Approximate request count in sliding window
     */
    public function getCount(string $id, int $window = 60): int
    {
        $now = time();

        $currentWindow = (int) floor($now / $window);
        $previousWindow = $currentWindow - 1;
        $windowProgress = ($now % $window) / $window;

        $currentKey = "rl:{$id}:{$currentWindow}";
        $previousKey = "rl:{$id}:{$previousWindow}";

        $previousCount = (int) (Cache::get($previousKey) ?? 0);
        $currentCount = (int) (Cache::get($currentKey) ?? 0);

        return (int) floor($previousCount * (1 - $windowProgress)) + $currentCount;
    }

    /**
     * Get remaining requests allowed
     *
     * @param string $id Unique identifier
     * @param int $max Maximum requests allowed
     * @param int $window Time window in seconds
     * @return int Remaining requests (0 if limit reached)
     */
    public function getRemaining(string $id, int $max = 100, int $window = 60): int
    {
        $count = $this->getCount($id, $window);
        return max(0, $max - $count);
    }

    /**
     * Get time until rate limit resets (seconds)
     *
     * @param string $id Unique identifier
     * @param int $window Time window in seconds
     * @return int Seconds until next window
     */
    public function getResetTime(string $id, int $window = 60): int
    {
        $now = time();
        $currentWindowEnd = (int) (ceil($now / $window) * $window);
        return $currentWindowEnd - $now;
    }

    /**
     * Reset rate limit for an identifier
     *
     * @param string $id Unique identifier
     * @param int $window Time window in seconds (needed to calculate keys)
     */
    public function reset(string $id, int $window = 60): void
    {
        $now = time();
        $currentWindow = (int) floor($now / $window);
        $previousWindow = $currentWindow - 1;

        Cache::delete("rl:{$id}:{$currentWindow}");
        Cache::delete("rl:{$id}:{$previousWindow}");
    }

    /**
     * Check rate limit and return detailed info
     *
     * @param string $id Unique identifier
     * @param int $max Maximum requests allowed
     * @param int $window Time window in seconds
     * @return array{allowed: bool, remaining: int, reset: int, count: int}
     */
    public function checkWithInfo(string $id, int $max = 100, int $window = 60): array
    {
        $allowed = $this->check($id, $max, $window);

        return [
            'allowed' => $allowed,
            'remaining' => $this->getRemaining($id, $max, $window),
            'reset' => $this->getResetTime($id, $window),
            'count' => $this->getCount($id, $window),
        ];
    }
}
