<?php

declare(strict_types=1);

namespace App;

/**
 * NDJSON Logger for redirect events
 * IP hashed for privacy
 */
final class RedirectLogger
{
    private string $logFile;

    public function __construct(string $logFile = '')
    {
        if ($logFile === '') {
            $logDir = __DIR__ . '/../data/logs';
            if (!is_dir($logDir)) {
                mkdir($logDir, 0755, true);
            }
            $this->logFile = $logDir . '/redirect.log';
        } else {
            $this->logFile = $logFile;
        }
    }

    /**
     * Log redirect event
     *
     * @param array<string,mixed> $data
     */
    public function log(array $data): void
    {
        $entry = [
            'timestamp' => date('c'),
            'event' => 'redirect',
        ];

        // Merge data
        $entry = array_merge($entry, $data);

        // Hash IP if present
        if (isset($entry['ip']) && is_string($entry['ip'])) {
            $entry['ip_hash'] = IpResolver::hash($entry['ip']);
            unset($entry['ip']); // Remove plaintext IP
        }

        // Encode as NDJSON (one JSON object per line)
        $json = json_encode($entry, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

        if ($json === false) {
            return; // Encoding failed
        }

        // Atomic write with file lock
        $fp = fopen($this->logFile, 'a');
        if ($fp === false) {
            return;
        }

        if (flock($fp, LOCK_EX)) {
            fwrite($fp, $json . "\n");
            flock($fp, LOCK_UN);
        }

        fclose($fp);
    }

    /**
     * Log blocked request
     *
     * @param array<string,mixed> $context
     */
    public function logBlocked(string $code, string $ip, string $reason, array $context = []): void
    {
        $this->log([
            'event' => 'blocked',
            'code' => $code,
            'ip' => $ip,
            'reason' => $reason,
            'context' => $context,
        ]);
    }

    /**
     * Log successful redirect
     *
     * @param array<string,mixed> $riskData
     */
    public function logRedirect(
        string $code,
        string $ip,
        string $targetUrl,
        string $routeType,
        array $riskData = []
    ): void {
        $this->log([
            'event' => 'redirect',
            'code' => $code,
            'ip' => $ip,
            'target_url' => $targetUrl,
            'route_type' => $routeType, // primary, safe, fallback
            'risk_score' => $riskData['risk_score'] ?? 0,
            'is_risky' => $riskData['is_risky'] ?? false,
            'country' => $riskData['country'] ?? '',
            'asn' => $riskData['asn'] ?? 0,
        ]);
    }
}
