<?php

declare(strict_types=1);

namespace App;

/**
 * Redirect Router - Decision maker
 * Routes visitor based on risk assessment
 */
final class RedirectRouter
{
    private RiskScorer $riskScorer;
    private RedirectLogger $logger;
    private bool $enableIpIntel;
    private string $primaryUrl;
    private string $safeUrl;
    private string $fallbackUrl;

    public function __construct(
        string $primaryUrl,
        string $safeUrl = '',
        string $fallbackUrl = '',
        bool $enableIpIntel = false
    ) {
        $this->riskScorer = new RiskScorer();
        $this->logger = new RedirectLogger();
        $this->enableIpIntel = $enableIpIntel;
        $this->primaryUrl = $primaryUrl;
        $this->safeUrl = $safeUrl !== '' ? $safeUrl : $primaryUrl;
        $this->fallbackUrl = $fallbackUrl !== '' ? $fallbackUrl : $safeUrl;
    }

    /**
     * Decide redirect target based on IP risk
     *
     * @return array{url:string,route_type:string,risk_data:array<string,mixed>}
     */
    public function route(string $ip, string $code): array
    {
        // If IP intel disabled, always use primary
        if (!$this->enableIpIntel) {
            return [
                'url' => $this->primaryUrl,
                'route_type' => 'primary',
                'risk_data' => ['is_risky' => false, 'risk_score' => 0],
            ];
        }

        // Assess risk
        try {
            $riskData = $this->riskScorer->assess($ip);

            if ($riskData['is_risky']) {
                // Risky IP - route to safe URL
                $this->logger->logRedirect($code, $ip, $this->safeUrl, 'safe', $riskData);

                return [
                    'url' => $this->safeUrl,
                    'route_type' => 'safe',
                    'risk_data' => $riskData,
                ];
            }

            // Clean IP - route to primary URL
            $this->logger->logRedirect($code, $ip, $this->primaryUrl, 'primary', $riskData);

            return [
                'url' => $this->primaryUrl,
                'route_type' => 'primary',
                'risk_data' => $riskData,
            ];
        } catch (\Throwable $e) {
            // Intel failed - use fallback
            error_log('Redirect router error: ' . $e->getMessage());

            $this->logger->logRedirect($code, $ip, $this->fallbackUrl, 'fallback', [
                'error' => $e->getMessage(),
            ]);

            return [
                'url' => $this->fallbackUrl,
                'route_type' => 'fallback',
                'risk_data' => ['is_risky' => false, 'risk_score' => 0, 'error' => true],
            ];
        }
    }

    /**
     * Log blocked request
     *
     * @param array<string,mixed> $context
     */
    public function logBlocked(string $code, string $ip, string $reason, array $context = []): void
    {
        $this->logger->logBlocked($code, $ip, $reason, $context);
    }
}
