<?php

declare(strict_types=1);

namespace App;

use App\IpIntel\AtlasClient;
use App\IpIntel\BlackboxClient;
use App\IpIntel\CurlHttpClient;
use App\IpIntel\IpQueryClient;

/**
 * Risk Scorer - Aggregate IP intel and calculate risk
 */
final class RiskScorer
{
    private BlackboxClient $blackbox;
    private IpQueryClient $ipquery;
    private AtlasClient $atlas;
    private int $riskThreshold;

    public function __construct(int $riskThreshold = 75)
    {
        $http = new CurlHttpClient();

        $this->blackbox = new BlackboxClient($http);
        $this->ipquery = new IpQueryClient($http);
        $this->atlas = new AtlasClient($http);
        $this->riskThreshold = $riskThreshold;
    }

    /**
     * Assess IP risk
     *
     * @return array{
     *   is_risky:bool,
     *   risk_score:int,
     *   reasons:array<string>,
     *   blackbox_signal:string,
     *   is_vpn:bool,
     *   is_proxy:bool,
     *   is_tor:bool,
     *   is_datacenter:bool,
     *   country:string,
     *   asn:int
     * }
     */
    public function assess(string $ip): array
    {
        $riskScore = 0;
        $reasons = [];

        // Check Blackbox first (fastest)
        $blackboxSignal = $this->blackbox->check($ip);

        if ($blackboxSignal === 'Y') {
            $riskScore += 100; // Auto-risky
            $reasons[] = 'Blacklisted';
        }

        // Query IPQuery for detailed intel
        $ipqueryResult = $this->ipquery->query($ip);

        $isVpn = false;
        $isProxy = false;
        $isTor = false;
        $isDatacenter = false;
        $country = '';
        $asn = 0;

        if ($ipqueryResult['success']) {
            $flags = $ipqueryResult['flags'];
            $isVpn = $flags['is_vpn'] ?? false;
            $isProxy = $flags['is_proxy'] ?? false;
            $isTor = $flags['is_tor'] ?? false;
            $isDatacenter = $flags['is_datacenter'] ?? false;
            $country = $ipqueryResult['country_code'] ?? '';
            $asn = $ipqueryResult['asn'] ?? 0;

            // Add risk points
            if ($isTor) {
                $riskScore += 80;
                $reasons[] = 'Tor exit node';
            }

            if ($isVpn) {
                $riskScore += 40;
                $reasons[] = 'VPN detected';
            }

            if ($isProxy) {
                $riskScore += 50;
                $reasons[] = 'Proxy detected';
            }

            if ($isDatacenter) {
                $riskScore += 30;
                $reasons[] = 'Datacenter IP';
            }

            // Use provided risk score if available
            $baseRisk = $ipqueryResult['base_risk_score'];
            if ($baseRisk > 0) {
                $riskScore = max($riskScore, $baseRisk);
            }
        } else {
            // IPQuery failed - try Atlas fallback
            $atlasResult = $this->atlas->lookup($ip);
            if ($atlasResult['success']) {
                $country = $atlasResult['country_code'] ?? '';
                $asn = $atlasResult['asn'] ?? 0;
            }
        }

        // Cap risk score at 100
        $riskScore = min($riskScore, 100);

        return [
            'is_risky' => $riskScore >= $this->riskThreshold,
            'risk_score' => $riskScore,
            'reasons' => $reasons,
            'blackbox_signal' => $blackboxSignal,
            'is_vpn' => $isVpn,
            'is_proxy' => $isProxy,
            'is_tor' => $isTor,
            'is_datacenter' => $isDatacenter,
            'country' => $country,
            'asn' => $asn,
        ];
    }

    /**
     * Quick check - only Blackbox (fastest)
     */
    public function quickCheck(string $ip): bool
    {
        $signal = $this->blackbox->check($ip);
        return $signal === 'Y';
    }
}
