<?php

declare(strict_types=1);

namespace App;

final class SecurityConfig
{
    private static ?array $config = null;

    /**
     * Load security configuration from file
     *
     * @return array<string,mixed>
     */
    public static function load(): array
    {
        if (self::$config !== null) {
            return self::$config;
        }

        $configFile = __DIR__ . '/../data/security-config.json';

        $defaults = [
            'cpanel_host' => '',
            'cpanel_username' => '',
            'cpanel_token' => '',
            'cf_api_token' => '',
            'cf_default_proxied' => false,
            'cf_default_ttl' => 1,
            'rate_limit_enabled' => false,
            'rate_limit_max_requests' => 100,
            'rate_limit_window' => 60,
            'rate_limit_by' => 'ip',
            'rate_limit_whitelist_admin' => true,
        ];

        if (!file_exists($configFile)) {
            self::$config = $defaults;
            return self::$config;
        }

        $loaded = json_decode(file_get_contents($configFile), true);
        if (!is_array($loaded)) {
            self::$config = $defaults;
            return self::$config;
        }

        self::$config = array_merge($defaults, $loaded);
        return self::$config;
    }

    public static function get(string $key, mixed $default = null): mixed
    {
        $config = self::load();
        return $config[$key] ?? $default;
    }

    public static function getCpanelHost(): string
    {
        return (string) self::get('cpanel_host', '');
    }

    public static function getCpanelUsername(): string
    {
        return (string) self::get('cpanel_username', '');
    }

    public static function getCpanelToken(): string
    {
        return (string) self::get('cpanel_token', '');
    }

    public static function getCloudflareApiToken(): string
    {
        return (string) self::get('cf_api_token', '');
    }

    public static function getCloudflareDefaultProxied(): bool
    {
        return (bool) self::get('cf_default_proxied', false);
    }

    public static function getCloudflareDefaultTtl(): int
    {
        return (int) self::get('cf_default_ttl', 1);
    }

    /**
     * Check if cPanel is configured
     */
    public static function hasCpanelConfig(): bool
    {
        return self::getCpanelHost() !== ''
            && self::getCpanelUsername() !== ''
            && self::getCpanelToken() !== '';
    }

    /**
     * Check if Cloudflare is configured
     */
    public static function hasCloudflareConfig(): bool
    {
        return self::getCloudflareApiToken() !== '';
    }

    /**
     * Check if rate limiting is enabled
     */
    public static function isRateLimitEnabled(): bool
    {
        return (bool) self::get('rate_limit_enabled', false);
    }

    /**
     * Get maximum requests allowed per window
     */
    public static function getRateLimitMaxRequests(): int
    {
        return (int) self::get('rate_limit_max_requests', 100);
    }

    /**
     * Get rate limit time window in seconds
     */
    public static function getRateLimitWindow(): int
    {
        return (int) self::get('rate_limit_window', 60);
    }

    /**
     * Get rate limit identifier type (ip, user, ip_user)
     */
    public static function getRateLimitBy(): string
    {
        $by = (string) self::get('rate_limit_by', 'ip');
        return in_array($by, ['ip', 'user', 'ip_user'], true) ? $by : 'ip';
    }

    /**
     * Check if admin users are whitelisted from rate limiting
     */
    public static function isRateLimitWhitelistAdmin(): bool
    {
        return (bool) self::get('rate_limit_whitelist_admin', true);
    }
}
