<?php

declare(strict_types=1);

namespace App;

/**
 * ShimLink - Generate signed shim links for safe redirect
 *
 * Usage:
 *   $shimUrl = ShimLink::generate('https://example.com/target');
 *   header('Location: ' . $shimUrl);
 */
final class ShimLink
{
    /**
     * Generate shim link for destination URL
     *
     * @param string $destinationUrl Target URL to wrap in shim
     * @return string Full shim URL with signature
     * @throws \RuntimeException If SHIM_HMAC_SECRET or SHIM_BASE_URL not configured
     */
    public static function generate(string $destinationUrl): string
    {
        $secret = Env::getString('SHIM_HMAC_SECRET', '');
        $baseUrl = Env::getString('SHIM_BASE_URL', '');

        if ($secret === '') {
            throw new \RuntimeException('SHIM_HMAC_SECRET not configured');
        }

        if ($baseUrl === '') {
            throw new \RuntimeException('SHIM_BASE_URL not configured');
        }

        // Base64url encode destination
        $d = Base64Url::encode($destinationUrl);

        // Generate HMAC signature (base64url encoded)
        $sigRaw = hash_hmac('sha256', $d, $secret, true);
        $sig = Base64Url::encode($sigRaw);

        // Build shim URL (no rawurlencode needed, base64url is URL-safe)
        $sep = str_contains($baseUrl, '?') ? '&' : '?';
        return $baseUrl . $sep . 'd=' . $d . '&sig=' . $sig;
    }

    /**
     * Check if shim link is enabled
     *
     * @return bool
     */
    public static function isEnabled(): bool
    {
        return Env::getBool('SHIM_ENABLED', false);
    }
}
