<?php

declare(strict_types=1);

namespace App;

use PDO;

final class ShortlinkRepository
{
    public function __construct(private readonly PDO $pdo)
    {
    }

    /**
     * @return array{
     *   code:string,
     *   og_title:string,
     *   og_description:string,
     *   og_image_url:string,
     *   og_type:string,
     *   og_site_name:string,
     *   og_locale:string,
     *   use_shim:int
     * }|null
     */
    public function findActive(string $code): ?array
    {
        $sql = 'SELECT code, og_title, og_description, og_image_url, og_type, og_site_name, og_locale, use_shim
                FROM shortlinks
                WHERE code = :code AND is_active = 1
                LIMIT 1';
        $st = $this->pdo->prepare($sql);
        $st->execute(['code' => $code]);

        $row = $st->fetch();
        if (!is_array($row)) {
            return null;
        }

        /** @var array{code:string,og_title:string,og_description:string,og_image_url:string,og_type:string,og_site_name:string,og_locale:string,use_shim:int} $row */
        return $row;
    }

    /**
     * @return array{
     *   code:string,
     *   og_title:string,
     *   og_description:string,
     *   og_image_url:string,
     *   og_type:string,
     *   og_site_name:string,
     *   og_locale:string,
     *   is_active:int,
     *   use_shim:int,
     *   created_at:string,
     *   updated_at:string
     * }|null
     */
    public function findByCode(string $code): ?array
    {
        $sql = 'SELECT code, og_title, og_description, og_image_url, og_type, og_site_name, og_locale,
                       is_active, use_shim, created_at, updated_at
                FROM shortlinks
                WHERE code = :code
                LIMIT 1';
        $st = $this->pdo->prepare($sql);
        $st->execute(['code' => $code]);

        $row = $st->fetch();
        if (!is_array($row)) {
            return null;
        }

        /** @var array{code:string,og_title:string,og_description:string,og_image_url:string,og_type:string,og_site_name:string,og_locale:string,is_active:int,use_shim:int,created_at:string,updated_at:string} $row */
        return $row;
    }

    /**
     * @param array{
     *   code:string,
     *   og_title:string,
     *   og_description:string,
     *   og_image_url:string,
     *   og_type:string,
     *   og_site_name:string,
     *   og_locale:string,
     *   use_shim:int,
     *   domain?:string,
     *   domain_type?:string,
     *   domain_id?:int
     * } $data
     */
    public function insert(array $data): void
    {
        $sql = 'INSERT INTO shortlinks
          (code, og_title, og_description, og_image_url, og_type, og_site_name, og_locale, is_active, use_shim, domain, domain_type, domain_id)
          VALUES
          (:code, :og_title, :og_description, :og_image_url, :og_type, :og_site_name, :og_locale, 1, :use_shim, :domain, :domain_type, :domain_id)';
        $st = $this->pdo->prepare($sql);
        $st->execute([
            'code' => $data['code'],
            'og_title' => $data['og_title'],
            'og_description' => $data['og_description'],
            'og_image_url' => $data['og_image_url'],
            'og_type' => $data['og_type'],
            'og_site_name' => $data['og_site_name'],
            'og_locale' => $data['og_locale'],
            'use_shim' => $data['use_shim'],
            'domain' => $data['domain'] ?? null,
            'domain_type' => $data['domain_type'] ?? null,
            'domain_id' => $data['domain_id'] ?? null,
        ]);
    }

    public function exists(string $code): bool
    {
        $sql = 'SELECT 1 FROM shortlinks WHERE code = :code LIMIT 1';
        $st  = $this->pdo->prepare($sql);
        $st->execute(['code' => $code]);

        return $st->fetchColumn() !== false;
    }

    /**
     * @return array{total:int, rows:list<array{
     *   code:string,
     *   og_title:string,
     *   og_description:string,
     *   og_image_url:string,
     *   og_type:string,
     *   og_site_name:string,
     *   og_locale:string,
     *   is_active:int,
     *   use_shim:int,
     *   created_at:string,
     *   updated_at:string
     * }>}
     */
    public function list(string $q, ?int $isActive, int $limit, int $offset): array
    {
        if ($limit < 1) {
            $limit = 20;
        }
        if ($limit > 100) {
            $limit = 100;
        }
        if ($offset < 0) {
            $offset = 0;
        }

        $where = ' WHERE 1=1';
        $params = [];

        if ($q !== '') {
            $where .= ' AND (s.code LIKE :like OR s.og_title LIKE :like)';
            $params['like'] = '%' . $q . '%';
        }

        if ($isActive !== null) {
            $where .= ' AND s.is_active = :active';
            $params['active'] = $isActive;
        }

        $sqlCount = 'SELECT COUNT(*) FROM shortlinks s' . $where;
        $stCount = $this->pdo->prepare($sqlCount);
        foreach ($params as $k => $v) {
            if ($k === 'active') {
                $stCount->bindValue($k, (int) $v, PDO::PARAM_INT);
            } else {
                $stCount->bindValue($k, (string) $v, PDO::PARAM_STR);
            }
        }
        $stCount->execute();
        $total = (int) $stCount->fetchColumn();

        $sql = 'SELECT s.code, s.og_title, s.og_description, s.og_image_url, s.og_type, s.og_site_name, s.og_locale,
                       s.is_active, s.use_shim, s.domain, s.created_at, s.updated_at
                FROM shortlinks s'
            . $where
            . ' ORDER BY s.updated_at DESC, s.code DESC'
            . ' LIMIT :limit OFFSET :offset';

        $st = $this->pdo->prepare($sql);
        foreach ($params as $k => $v) {
            if ($k === 'active') {
                $st->bindValue($k, (int) $v, PDO::PARAM_INT);
            } else {
                $st->bindValue($k, (string) $v, PDO::PARAM_STR);
            }
        }
        $st->bindValue('limit', $limit, PDO::PARAM_INT);
        $st->bindValue('offset', $offset, PDO::PARAM_INT);
        $st->execute();

        $rows = $st->fetchAll();
        if (!is_array($rows)) {
            $rows = [];
        }

        /** @var list<array{code:string,og_title:string,og_description:string,og_image_url:string,og_type:string,og_site_name:string,og_locale:string,is_active:int,use_shim:int,created_at:string,updated_at:string}> $rows */
        return ['total' => $total, 'rows' => $rows];
    }

    /**
     * @param array{
     *   og_title:string,
     *   og_description:string,
     *   og_image_url:string,
     *   og_type:string,
     *   og_site_name:string,
     *   og_locale:string,
     *   use_shim:int,
     *   domain?:string,
     *   domain_type?:string,
     *   domain_id?:int
     * } $data
     */
    public function updateByCode(string $code, array $data): void
    {
        $sql = 'UPDATE shortlinks
                SET og_title = :og_title,
                    og_description = :og_description,
                    og_image_url = :og_image_url,
                    og_type = :og_type,
                    og_site_name = :og_site_name,
                    og_locale = :og_locale,
                    use_shim = :use_shim,
                    domain = :domain,
                    domain_type = :domain_type,
                    domain_id = :domain_id,
                    updated_at = NOW()
                WHERE code = :code
                LIMIT 1';

        $st = $this->pdo->prepare($sql);
        $st->execute([
            'code' => $code,
            'og_title' => $data['og_title'],
            'og_description' => $data['og_description'],
            'og_image_url' => $data['og_image_url'],
            'og_type' => $data['og_type'],
            'og_site_name' => $data['og_site_name'],
            'og_locale' => $data['og_locale'],
            'use_shim' => $data['use_shim'],
            'domain' => $data['domain'] ?? null,
            'domain_type' => $data['domain_type'] ?? null,
            'domain_id' => $data['domain_id'] ?? null,
        ]);
    }

    public function setActive(string $code, int $isActive): void
    {
        $sql = 'UPDATE shortlinks SET is_active = :a, updated_at = NOW() WHERE code = :code LIMIT 1';
        $st = $this->pdo->prepare($sql);
        $st->bindValue('a', $isActive, PDO::PARAM_INT);
        $st->bindValue('code', $code, PDO::PARAM_STR);
        $st->execute();
    }
}
