<?php

declare(strict_types=1);

namespace App;

/**
 * Signed URL - Anti enumeration + anti link forging
 * Uses HMAC-SHA256 with TTL
 */
final class SignedUrl
{
    private const ALGO = 'sha256';

    /**
     * Generate signed URL
     *
     * @param string $code Shortlink code
     * @param int $ttl Time to live in seconds (0 = no expiry)
     * @param string $secret Secret key for signing
     * @return string Full URL with signature and expiry
     */
    public static function generate(string $code, int $ttl = 3600, string $secret = ''): string
    {
        if ($secret === '') {
            $secret = Env::getString('URL_SIGN_SECRET', 'change_this_secret');
        }

        $expires = $ttl > 0 ? (time() + $ttl) : 0;

        // Payload: code + expiry
        $payload = $code . '|' . $expires;

        // Generate HMAC
        $signature = hash_hmac(self::ALGO, $payload, $secret);

        // Build URL
        $baseUrl = Env::getString('PUBLIC_BASE_URL', 'http://localhost:8000');
        $baseUrl = rtrim($baseUrl, '/');

        $query = http_build_query([
            'code' => $code,
            'expires' => $expires,
            'sig' => $signature,
        ]);

        return $baseUrl . '/?' . $query;
    }

    /**
     * Verify signed URL
     *
     * @param array<string,mixed> $params
     * @return array{valid:bool,error:string,code:string}
     */
    public static function verify(array $params, string $secret = ''): array
    {
        if ($secret === '') {
            $secret = Env::getString('URL_SIGN_SECRET', 'change_this_secret');
        }

        $codeParam = $params['code'] ?? '';
        $expiresParam = $params['expires'] ?? 0;
        $sigParam = $params['sig'] ?? '';

        $code = is_string($codeParam) ? $codeParam : '';
        $expires = is_int($expiresParam) ? $expiresParam : (is_string($expiresParam) ? (int) $expiresParam : 0);
        $signature = is_string($sigParam) ? $sigParam : '';

        if ($code === '' || $signature === '') {
            return ['valid' => false, 'error' => 'Missing parameters', 'code' => ''];
        }

        // Check expiry
        if ($expires > 0 && time() > $expires) {
            return ['valid' => false, 'error' => 'URL expired', 'code' => ''];
        }

        // Verify signature
        $payload = $code . '|' . $expires;
        $expectedSignature = hash_hmac(self::ALGO, $payload, $secret);

        if (!hash_equals($expectedSignature, $signature)) {
            return ['valid' => false, 'error' => 'Invalid signature', 'code' => ''];
        }

        return ['valid' => true, 'error' => '', 'code' => $code];
    }

    /**
     * Check if URL signing is enabled
     */
    public static function isEnabled(): bool
    {
        return Env::getString('URL_SIGN_ENABLED', 'false') === 'true';
    }
}
