<?php

declare(strict_types=1);

namespace App;

use App\Infrastructure\Persistence\AbstractDomainRepository;
use PDO;

/**
 * Repository for user_domains table
 *
 * Manages per-user domain configurations for shortlink service
 */
final class UserDomainRepository extends AbstractDomainRepository
{
    protected string $tableName = 'user_domains';

    public function __construct(PDO $pdo)
    {
        parent::__construct($pdo);
    }

    /**
     * Create new user domain
     *
     * @param array{
     *   user_id: int,
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   cpanel_host?: string,
     *   cpanel_username?: string,
     *   cpanel_token?: string,
     *   cf_zone_id?: string,
     *   cf_api_token?: string,
     *   cf_proxied?: int,
     *   target_ip?: string,
     *   target_cname?: string
     * } $data
     * @return int Inserted ID
     */
    public function create(array $data): int
    {
        $sql = 'INSERT INTO user_domains
                (user_id, domain, provider, is_wildcard, cpanel_host, cpanel_username, cpanel_token,
                 cf_zone_id, cf_api_token, cf_proxied, target_ip, target_cname)
                VALUES
                (:user_id, :domain, :provider, :is_wildcard, :cpanel_host, :cpanel_username, :cpanel_token,
                 :cf_zone_id, :cf_api_token, :cf_proxied, :target_ip, :target_cname)';

        $params = $this->getCommonInsertParams($data);
        $params['user_id'] = $data['user_id'];

        return $this->executeInsert($sql, $params);
    }

    /**
     * List domains for specific user
     *
     * @param int $userId
     * @param bool|null $isActive Filter by active status (null = all)
     * @return list<array{
     *   id: int,
     *   user_id: int,
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   is_active: int,
     *   target_ip: string|null,
     *   target_cname: string|null,
     *   created_at: string,
     *   updated_at: string
     * }>
     */
    public function listByUser(int $userId, ?bool $isActive = null): array
    {
        $sql = 'SELECT id, user_id, domain, provider, is_wildcard, is_active,
                       target_ip, target_cname, created_at, updated_at
                FROM user_domains
                WHERE user_id = :user_id';

        $params = ['user_id' => $userId];

        if ($isActive !== null) {
            $sql .= ' AND is_active = :is_active';
            $params['is_active'] = $isActive ? 1 : 0;
        }

        $sql .= ' ORDER BY created_at DESC';

        /** @var list<array{id: int, user_id: int, domain: string, provider: string, is_wildcard: int, is_active: int, target_ip: string|null, target_cname: string|null, created_at: string, updated_at: string}> */
        return $this->fetchAll($sql, $params);
    }

    /**
     * List active domains for user (for dropdown)
     *
     * @param int $userId
     * @return list<array{id: int, domain: string, is_wildcard: int}>
     */
    public function listActiveByUser(int $userId): array
    {
        $sql = 'SELECT id, domain, is_wildcard
                FROM user_domains
                WHERE user_id = :user_id AND is_active = 1
                ORDER BY domain ASC';

        /** @var list<array{id: int, domain: string, is_wildcard: int}> */
        return $this->fetchAll($sql, ['user_id' => $userId]);
    }

    /**
     * Get domain by ID (with user ownership check)
     *
     * @param int $id Domain ID
     * @param int $userId Owner user ID
     * @return array{
     *   id: int,
     *   user_id: int,
     *   domain: string,
     *   provider: string,
     *   is_wildcard: int,
     *   is_active: int,
     *   cpanel_host: string|null,
     *   cpanel_username: string|null,
     *   cpanel_token: string|null,
     *   cf_zone_id: string|null,
     *   cf_api_token: string|null,
     *   cf_proxied: int,
     *   dns_record_id: string|null,
     *   target_ip: string|null,
     *   target_cname: string|null,
     *   last_sync_at: string|null,
     *   created_at: string,
     *   updated_at: string
     * }|null
     */
    public function findById(int $id, int $userId): ?array
    {
        $sql = 'SELECT *
                FROM user_domains
                WHERE id = :id AND user_id = :user_id
                LIMIT 1';

        /** @var array{id: int, user_id: int, domain: string, provider: string, is_wildcard: int, is_active: int, cpanel_host: string|null, cpanel_username: string|null, cpanel_token: string|null, cf_zone_id: string|null, cf_api_token: string|null, cf_proxied: int, dns_record_id: string|null, target_ip: string|null, target_cname: string|null, last_sync_at: string|null, created_at: string, updated_at: string}|null */
        return $this->fetchOne($sql, ['id' => $id, 'user_id' => $userId]);
    }

    /**
     * Get domain by ID without ownership check (admin use)
     *
     * @param int $id Domain ID
     * @return array<string, mixed>|null
     */
    public function findByIdAdmin(int $id): ?array
    {
        $sql = 'SELECT * FROM user_domains WHERE id = :id LIMIT 1';
        return $this->fetchOne($sql, ['id' => $id]);
    }

    /**
     * Update domain
     *
     * @param int $id Domain ID
     * @param int $userId Owner user ID
     * @param array<string, mixed> $data Fields to update
     * @return bool Success
     */
    public function update(int $id, int $userId, array $data): bool
    {
        $updateData = $this->buildUpdateQuery($data, ['id' => $id, 'user_id' => $userId]);
        if ($updateData === null) {
            return false;
        }

        $sql = 'UPDATE user_domains SET ' . $updateData['sql'] . '
                WHERE id = :id AND user_id = :user_id
                LIMIT 1';

        return $this->executeUpdate($sql, $updateData['params']);
    }

    /**
     * Update domain without ownership check (admin use)
     *
     * @param int $id Domain ID
     * @param array<string, mixed> $data Fields to update
     * @return bool Success
     */
    public function updateAdmin(int $id, array $data): bool
    {
        $updateData = $this->buildUpdateQuery($data, ['id' => $id]);
        if ($updateData === null) {
            return false;
        }

        $sql = 'UPDATE user_domains SET ' . $updateData['sql'] . '
                WHERE id = :id LIMIT 1';

        return $this->executeUpdate($sql, $updateData['params']);
    }

    /**
     * Delete domain
     *
     * @param int $id Domain ID
     * @param int $userId Owner user ID
     * @return bool Success
     */
    public function delete(int $id, int $userId): bool
    {
        $sql = 'DELETE FROM user_domains
                WHERE id = :id AND user_id = :user_id
                LIMIT 1';

        return $this->executeDelete($sql, ['id' => $id, 'user_id' => $userId]);
    }

    /**
     * Delete domain without ownership check (admin use)
     *
     * @param int $id Domain ID
     * @return bool Success
     */
    public function deleteAdmin(int $id): bool
    {
        $sql = 'DELETE FROM user_domains WHERE id = :id LIMIT 1';
        return $this->executeDelete($sql, ['id' => $id]);
    }

    /**
     * Check if domain exists for user
     *
     * @param string $domain
     * @param int $userId
     * @return bool
     */
    public function exists(string $domain, int $userId): bool
    {
        $sql = 'SELECT 1 FROM user_domains
                WHERE domain = :domain AND user_id = :user_id
                LIMIT 1';

        return $this->rowExists($sql, ['domain' => $domain, 'user_id' => $userId]);
    }

    /**
     * Check if domain exists globally (any user)
     *
     * @param string $domain
     * @return bool
     */
    public function existsGlobally(string $domain): bool
    {
        $sql = 'SELECT 1 FROM user_domains WHERE domain = :domain LIMIT 1';
        return $this->rowExists($sql, ['domain' => $domain]);
    }
}
