#!/usr/bin/env php
<?php

declare(strict_types=1);

/**
 * OG Shortlink Production Build Script
 *
 * Creates a production-ready ZIP package for cPanel deployment.
 *
 * Usage:
 *   php bin/build-release.php [version]
 *
 * Example:
 *   php bin/build-release.php 1.0.0
 */

$version = $argv[1] ?? '1.0.0';
$basePath = dirname(__DIR__);
$buildDir = $basePath . '/build';
$distDir = $basePath . '/dist';

echo "=== OG Shortlink Build Script ===\n";
echo "Version: {$version}\n";
echo "Base path: {$basePath}\n\n";

// Create build directories
if (!is_dir($buildDir)) {
    mkdir($buildDir, 0755, true);
}
if (!is_dir($distDir)) {
    mkdir($distDir, 0755, true);
}

// Clean previous build
$tempDir = $buildDir . '/og-shortlink';
if (is_dir($tempDir)) {
    deleteDirectory($tempDir);
}
mkdir($tempDir, 0755, true);

// Files and directories to include
$includes = [
    'app',
    'bin/traffic-worker.php',
    'config',
    'database',
    'install',
    'public',
    'src',
    'vendor',
    'composer.json',
    'composer.lock',
];

// Files and directories to exclude
$excludes = [
    '.git',
    '.gitignore',
    '.env',
    '.env.example',
    'build',
    'dist',
    'tests',
    'phpunit.xml',
    'phpstan.neon',
    'phpcs.xml',
    '.php-cs-fixer.php',
    'node_modules',
    '.idea',
    '.vscode',
    '*.log',
    'data/cache/*',
    'data/logs/*',
    '.installed',
];

echo "Copying files...\n";

foreach ($includes as $item) {
    $source = $basePath . '/' . $item;
    $dest = $tempDir . '/' . $item;

    if (!file_exists($source)) {
        echo "  [SKIP] {$item} (not found)\n";
        continue;
    }

    if (is_dir($source)) {
        copyDirectory($source, $dest, $excludes);
        echo "  [DIR]  {$item}\n";
    } else {
        $destDir = dirname($dest);
        if (!is_dir($destDir)) {
            mkdir($destDir, 0755, true);
        }
        copy($source, $dest);
        echo "  [FILE] {$item}\n";
    }
}

// Create empty directories
$emptyDirs = [
    'data',
    'data/cache',
    'data/logs',
];

foreach ($emptyDirs as $dir) {
    $path = $tempDir . '/' . $dir;
    if (!is_dir($path)) {
        mkdir($path, 0755, true);
    }
    // Add .gitkeep
    file_put_contents($path . '/.gitkeep', '');
    echo "  [EMPTY] {$dir}\n";
}

// Create .env.example
$envExample = <<<'ENV'
# OG Shortlink Configuration
# Copy this file to .env and configure your settings

# =============================================================================
# DATABASE
# =============================================================================
DB_DSN="mysql:host=localhost;port=3306;dbname=your_database;charset=utf8mb4"
DB_USER="your_username"
DB_PASS="your_password"

# =============================================================================
# APPLICATION
# =============================================================================
APP_ENV="production"
APP_DEBUG="false"
PUBLIC_BASE_URL="https://yourdomain.com"

# =============================================================================
# SECURITY (Generate with: php -r "echo bin2hex(random_bytes(32));")
# =============================================================================
SESSION_SECRET=""
ENCRYPTION_KEY=""

# =============================================================================
# SHORTLINK SETTINGS
# =============================================================================
SHORTLINK_API_KEY=""
SHORTLINK_HTTPS_ONLY="true"
SHORTLINK_BLOCK_IP_LITERAL="true"
SHORTLINK_BLOCK_USERINFO="true"

# URL Signing (optional)
URL_SIGN_ENABLED="false"
URL_SIGN_SECRET=""

# =============================================================================
# TRAFFIC LOGGING
# =============================================================================
TRAFFIC_LOG_ENABLED="true"
TRAFFIC_LOG_ASYNC="true"

# =============================================================================
# IP INTELLIGENCE (Optional)
# =============================================================================
IP_INTEL_ENABLED="false"
# IP_INTEL_PROVIDER="ipinfo"
# IPINFO_TOKEN=""

# =============================================================================
# SHIM LINK (Optional)
# =============================================================================
SHIM_ENABLED="false"
# SHIM_BASE_URL=""
# SHIM_SECRET=""
ENV;

file_put_contents($tempDir . '/.env.example', $envExample);
echo "  [FILE] .env.example\n";

// Create .htaccess for root
$htaccessRoot = <<<'HTACCESS'
# Redirect all requests to public folder
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteRule ^$ public/ [L]
    RewriteRule (.*) public/$1 [L]
</IfModule>

# Deny access to sensitive files
<FilesMatch "^\.">
    Order allow,deny
    Deny from all
</FilesMatch>

<FilesMatch "\.(env|json|lock|md|sql)$">
    Order allow,deny
    Deny from all
</FilesMatch>
HTACCESS;

file_put_contents($tempDir . '/.htaccess', $htaccessRoot);
echo "  [FILE] .htaccess\n";

// Create README for installation
$readme = <<<README
# OG Shortlink v{$version}

## Quick Installation (cPanel)

1. **Upload Files**
   - Upload all files to your hosting (public_html or subdomain folder)
   - Or extract the ZIP in File Manager

2. **Set Permissions**
   ```
   chmod 755 data data/cache data/logs config
   ```

3. **Run Installer**
   - Navigate to: https://yourdomain.com/install/
   - Follow the installation wizard

4. **Set Up Cron Jobs** (cPanel → Cron Jobs)
   ```
   * * * * * php /home/username/public_html/bin/traffic-worker.php
   ```

5. **Access Admin Panel**
   - URL: https://yourdomain.com/admin/
   - Login with credentials you created during installation

## Directory Structure

```
├── app/           # Core application classes
├── bin/           # CLI scripts
├── config/        # Configuration files
├── data/          # Runtime data (cache, logs)
├── database/      # SQL migrations
├── install/       # Installer (delete after setup)
├── public/        # Web root
├── src/           # Application source code
└── vendor/        # Composer dependencies
```

## Security Checklist

- [ ] SSL/HTTPS enabled
- [ ] .env file permissions set to 600
- [ ] install/ folder deleted
- [ ] Strong admin password set
- [ ] Regular database backups configured

## Support

For issues and feature requests:
https://github.com/your-repo/og-shortlink/issues

README;

file_put_contents($tempDir . '/README.md', $readme);
echo "  [FILE] README.md\n";

// Create version file
file_put_contents($tempDir . '/VERSION', $version);
echo "  [FILE] VERSION\n";

// Create ZIP archive
$zipFile = $distDir . '/og-shortlink-' . $version . '.zip';

echo "\nCreating ZIP archive...\n";

if (file_exists($zipFile)) {
    unlink($zipFile);
}

$zipCreated = false;

// Method 1: Try ZipArchive extension
if (class_exists('ZipArchive')) {
    echo "  Using: ZipArchive extension\n";
    $zip = new ZipArchive();
    if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
        addDirectoryToZip($zip, $tempDir, 'og-shortlink');
        $zip->close();
        $zipCreated = true;
    }
}

// Method 2: Try 7-Zip (Windows)
if (!$zipCreated && strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
    $sevenZipPaths = [
        'C:\\Program Files\\7-Zip\\7z.exe',
        'C:\\Program Files (x86)\\7-Zip\\7z.exe',
        '7z.exe', // If in PATH
    ];

    foreach ($sevenZipPaths as $sevenZip) {
        if (file_exists($sevenZip) || $sevenZip === '7z.exe') {
            $cmd = sprintf(
                '"%s" a -tzip "%s" "%s\\*" -r',
                $sevenZip,
                $zipFile,
                $tempDir
            );
            echo "  Using: 7-Zip\n";
            exec($cmd . ' 2>&1', $output, $returnCode);
            if ($returnCode === 0 && file_exists($zipFile)) {
                $zipCreated = true;
                break;
            }
        }
    }
}

// Method 3: Try PowerShell (Windows)
if (!$zipCreated && strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
    echo "  Using: PowerShell Compress-Archive\n";
    $cmd = sprintf(
        'powershell -Command "Compress-Archive -Path \"%s\\*\" -DestinationPath \"%s\" -Force"',
        $tempDir,
        $zipFile
    );
    exec($cmd . ' 2>&1', $output, $returnCode);
    if ($returnCode === 0 && file_exists($zipFile)) {
        $zipCreated = true;
    }
}

// Method 4: Try system zip command (Linux/Mac)
if (!$zipCreated && strtoupper(substr(PHP_OS, 0, 3)) !== 'WIN') {
    echo "  Using: zip command\n";
    $currentDir = getcwd();
    chdir($buildDir);
    $cmd = sprintf('zip -r "%s" "og-shortlink"', $zipFile);
    exec($cmd . ' 2>&1', $output, $returnCode);
    chdir($currentDir);
    if ($returnCode === 0 && file_exists($zipFile)) {
        $zipCreated = true;
    }
}

if ($zipCreated && file_exists($zipFile)) {
    $size = filesize($zipFile);
    $sizeFormatted = number_format($size / 1024 / 1024, 2);
    echo "  Created: {$zipFile}\n";
    echo "  Size: {$sizeFormatted} MB\n";

    // Cleanup temp directory
    echo "\nCleaning up...\n";
    deleteDirectory($tempDir);

    echo "\n=== Build Complete ===\n";
    echo "Output: {$zipFile}\n";
} else {
    echo "\n[WARNING] Could not create ZIP automatically.\n";
    echo "The build folder is ready at: {$tempDir}\n";
    echo "\nTo create ZIP manually:\n";
    echo "  - Windows: Right-click folder > Send to > Compressed (zipped) folder\n";
    echo "  - Or use 7-Zip/WinRAR to create ZIP\n";
    echo "\n=== Build Partial Complete ===\n";
    echo "Output folder: {$tempDir}\n";
}

// Helper functions

function copyDirectory(string $source, string $dest, array $excludes = []): void
{
    if (!is_dir($dest)) {
        mkdir($dest, 0755, true);
    }

    $iterator = new DirectoryIterator($source);
    foreach ($iterator as $item) {
        if ($item->isDot()) {
            continue;
        }

        $name = $item->getFilename();

        // Check excludes
        foreach ($excludes as $exclude) {
            if ($name === $exclude || fnmatch($exclude, $name)) {
                continue 2;
            }
        }

        $sourcePath = $item->getPathname();
        $destPath = $dest . '/' . $name;

        if ($item->isDir()) {
            copyDirectory($sourcePath, $destPath, $excludes);
        } else {
            copy($sourcePath, $destPath);
        }
    }
}

function deleteDirectory(string $dir): void
{
    if (!is_dir($dir)) {
        return;
    }

    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );

    foreach ($iterator as $item) {
        if ($item->isDir()) {
            rmdir($item->getPathname());
        } else {
            unlink($item->getPathname());
        }
    }

    rmdir($dir);
}

function addDirectoryToZip(ZipArchive $zip, string $dir, string $zipPath): void
{
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($iterator as $item) {
        $filePath = $item->getPathname();
        $relativePath = $zipPath . '/' . substr($filePath, strlen($dir) + 1);

        if ($item->isDir()) {
            $zip->addEmptyDir($relativePath);
        } else {
            $zip->addFile($filePath, $relativePath);
        }
    }
}
