#!/usr/bin/env bash

# OG Shortlink - cPanel Cron Job Setup Helper
# Usage: ./bin/setup-cron.sh

set -euo pipefail

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}=== OG Shortlink - cPanel Cron Setup ===${NC}\n"

# Detect project root
PROJECT_ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
echo -e "Project root: ${YELLOW}${PROJECT_ROOT}${NC}\n"

# Detect PHP binary
if command -v php83 &> /dev/null; then
    PHP_BIN="$(command -v php83)"
elif command -v php8.3 &> /dev/null; then
    PHP_BIN="$(command -v php8.3)"
elif command -v php &> /dev/null; then
    PHP_BIN="$(command -v php)"
else
    echo -e "${RED}Error: PHP not found. Please install PHP 8.3+${NC}"
    exit 1
fi

echo -e "PHP binary: ${YELLOW}${PHP_BIN}${NC}"

# Check PHP version
PHP_VERSION=$("${PHP_BIN}" -v | head -n 1 | cut -d ' ' -f 2 | cut -d '.' -f 1,2)
echo -e "PHP version: ${YELLOW}${PHP_VERSION}${NC}\n"

# Version comparison without bc (more portable)
version_compare() {
    local ver1="${1//./ }"
    local ver2="${2//./ }"
    local v1_arr=($ver1)
    local v2_arr=($ver2)

    # Compare major version
    if [ "${v1_arr[0]}" -lt "${v2_arr[0]}" ]; then
        return 0  # v1 < v2
    elif [ "${v1_arr[0]}" -gt "${v2_arr[0]}" ]; then
        return 1  # v1 >= v2
    fi

    # Compare minor version
    if [ "${v1_arr[1]}" -lt "${v2_arr[1]}" ]; then
        return 0  # v1 < v2
    else
        return 1  # v1 >= v2
    fi
}

if version_compare "${PHP_VERSION}" "8.3"; then
    echo -e "${RED}Error: PHP 8.3+ required. Current: ${PHP_VERSION}${NC}"
    exit 1
fi

# Generate cron commands
echo -e "${GREEN}Cron job commands (copy ke cPanel):${NC}\n"

echo -e "${YELLOW}1. Background Worker (Every minute):${NC}"
echo "${PHP_BIN} ${PROJECT_ROOT}/bin/worker.php >> ${PROJECT_ROOT}/var/logs/worker.log 2>&1"
echo ""

echo -e "${YELLOW}2. Cleanup Old Logs (Daily at 2 AM):${NC}"
echo "find ${PROJECT_ROOT}/var/logs -name \"*.log\" -mtime +7 -delete"
echo ""

echo -e "${YELLOW}3. Download SFS Blocklist (Daily at 3 AM):${NC}"
if [ -f "${PROJECT_ROOT}/bin/download_sfs_blocklist.php" ]; then
    echo "${PHP_BIN} ${PROJECT_ROOT}/bin/download_sfs_blocklist.php >> ${PROJECT_ROOT}/var/logs/sfs.log 2>&1"
else
    echo -e "${RED}[SKIP] File not found: bin/download_sfs_blocklist.php${NC}"
fi
echo ""

# Test worker script
echo -e "${GREEN}Testing worker script...${NC}"
if [ -f "${PROJECT_ROOT}/bin/worker.php" ]; then
    echo -e "${YELLOW}Running: ${PHP_BIN} ${PROJECT_ROOT}/bin/worker.php (timeout 5s)${NC}"
    timeout 5s "${PHP_BIN}" "${PROJECT_ROOT}/bin/worker.php" || true
    echo -e "${GREEN}✓ Worker script executed${NC}\n"
else
    echo -e "${RED}✗ Worker script not found: ${PROJECT_ROOT}/bin/worker.php${NC}\n"
fi

# Instructions
echo -e "${GREEN}=== Setup Instructions ===${NC}\n"
echo "1. Login to cPanel"
echo "2. Navigate to: Advanced → Cron Jobs"
echo "3. Add each cron job with the commands above"
echo ""
echo -e "${YELLOW}Cron Format Reference:${NC}"
echo "*/1 * * * *    = Every minute"
echo "0 2 * * *      = Daily at 2:00 AM"
echo "0 3 * * *      = Daily at 3:00 AM"
echo "0 */6 * * *    = Every 6 hours"
echo ""

# Verify directories
echo -e "${GREEN}=== Directory Check ===${NC}\n"

if [ -d "${PROJECT_ROOT}/var/logs" ]; then
    echo -e "${GREEN}✓${NC} var/logs exists"

    # Check writable
    if [ -w "${PROJECT_ROOT}/var/logs" ]; then
        echo -e "${GREEN}✓${NC} var/logs is writable"
    else
        echo -e "${RED}✗${NC} var/logs is NOT writable"
        echo -e "  Fix: chmod 755 ${PROJECT_ROOT}/var/logs"
    fi
else
    echo -e "${RED}✗${NC} var/logs does NOT exist"
    echo -e "  Creating: mkdir -p ${PROJECT_ROOT}/var/logs"
    mkdir -p "${PROJECT_ROOT}/var/logs"
    chmod 755 "${PROJECT_ROOT}/var/logs"
fi

if [ -d "${PROJECT_ROOT}/var/cache" ]; then
    echo -e "${GREEN}✓${NC} var/cache exists"
else
    echo -e "${YELLOW}!${NC} var/cache does NOT exist (creating...)"
    mkdir -p "${PROJECT_ROOT}/var/cache"
    chmod 755 "${PROJECT_ROOT}/var/cache"
fi

echo ""
echo -e "${GREEN}Setup helper completed!${NC}"
echo -e "Copy the cron commands above to cPanel → Cron Jobs\n"
