#!/usr/bin/env php
<?php

declare(strict_types=1);

/**
 * Redis Connection Test
 *
 * Tests Redis connectivity using both ext-redis and Predis
 */

echo "=== Redis Connection Test ===\n\n";

// Test 1: ext-redis (PHP extension)
echo "[1] Testing ext-redis (PHP extension)...\n";
if (!extension_loaded('redis')) {
    echo "    ❌ ext-redis NOT installed\n";
    echo "    ℹ️  Using Predis library instead (fallback)\n\n";
} else {
    try {
        $redis = new \Redis();
        $redis->connect('127.0.0.1', 6379, 2);
        $pong = $redis->ping();
        echo "    ✅ Connection: SUCCESS\n";
        echo "    ✅ PING: {$pong}\n\n";
        $redis->close();
    } catch (\Throwable $e) {
        echo "    ❌ ERROR: {$e->getMessage()}\n\n";
    }
}

// Test 2: Predis (Composer library)
echo "[2] Testing Predis (Composer library)...\n";
require __DIR__ . '/../vendor/autoload.php';

try {
    $predis = new \Predis\Client([
        'scheme' => 'tcp',
        'host'   => '127.0.0.1',
        'port'   => 6379,
        'timeout' => 2,
    ]);

    $pong = $predis->ping();
    echo "    ✅ Connection: SUCCESS\n";
    echo "    ✅ PING: {$pong}\n\n";

    // Test SET/GET
    $testKey = 'test:' . time();
    $predis->set($testKey, 'Hello from Predis!', 'EX', 10);
    $value = $predis->get($testKey);
    echo "    ✅ SET/GET: {$value}\n";

    // Get info
    $info = $predis->info('server');
    echo "    ℹ️  Redis Version: {$info['Server']['redis_version']}\n";
    echo "    ℹ️  Uptime: {$info['Server']['uptime_in_seconds']}s\n\n";

} catch (\Throwable $e) {
    echo "    ❌ ERROR: {$e->getMessage()}\n\n";
    exit(1);
}

echo "=== All Tests Passed ✅ ===\n";
