#!/usr/bin/env php
<?php

declare(strict_types=1);

/**
 * Traffic Queue Worker
 *
 * Processes queued traffic log entries in the background.
 *
 * Usage:
 *   php bin/traffic-worker.php              # Single run (for cron)
 *   php bin/traffic-worker.php --daemon     # Continuous daemon mode
 *   php bin/traffic-worker.php --status     # Show queue status
 *
 * Cron example (run every minute):
 *   * * * * * php /path/to/bin/traffic-worker.php >> /var/log/traffic-worker.log 2>&1
 */

require_once __DIR__ . '/../vendor/autoload.php';

use App\Analytics\AsyncTrafficLogger;
use App\Env;

// Bootstrap
Env::bootstrap();

// Parse arguments
$options = getopt('', ['daemon', 'status', 'batch:', 'sleep:']);
$isDaemon = isset($options['daemon']);
$showStatus = isset($options['status']);
$batchSize = isset($options['batch']) ? (int) $options['batch'] : 100;
$sleepMs = isset($options['sleep']) ? (int) $options['sleep'] : 1000;

$logger = new AsyncTrafficLogger();

// Status mode
if ($showStatus) {
    $size = $logger->getQueueSize();
    echo "Queue size: {$size}\n";
    exit(0);
}

// Single run mode
if (!$isDaemon) {
    $processed = $logger->process($batchSize);
    if ($processed > 0) {
        echo "[" . date('Y-m-d H:i:s') . "] Processed {$processed} entries\n";
    }
    exit(0);
}

// Daemon mode
echo "[" . date('Y-m-d H:i:s') . "] Traffic worker started (daemon mode)\n";
echo "Batch size: {$batchSize}, Sleep: {$sleepMs}ms\n";

// Handle signals for graceful shutdown
$running = true;

if (function_exists('pcntl_signal')) {
    pcntl_signal(SIGTERM, function () use (&$running) {
        echo "\n[" . date('Y-m-d H:i:s') . "] Received SIGTERM, shutting down...\n";
        $running = false;
    });

    pcntl_signal(SIGINT, function () use (&$running) {
        echo "\n[" . date('Y-m-d H:i:s') . "] Received SIGINT, shutting down...\n";
        $running = false;
    });
}

// Main loop
$totalProcessed = 0;
$startTime = time();

while ($running) {
    // Check signals
    if (function_exists('pcntl_signal_dispatch')) {
        pcntl_signal_dispatch();
    }

    // Process batch
    $processed = $logger->process($batchSize);
    $totalProcessed += $processed;

    if ($processed > 0) {
        echo "[" . date('Y-m-d H:i:s') . "] Processed {$processed} entries (total: {$totalProcessed})\n";
    }

    // Sleep if no work or after processing
    usleep($sleepMs * 1000);

    // Reconnect database periodically (every 5 minutes) to prevent stale connections
    if ((time() - $startTime) > 300) {
        \App\Db::close();
        $startTime = time();
    }
}

echo "[" . date('Y-m-d H:i:s') . "] Worker stopped. Total processed: {$totalProcessed}\n";
exit(0);
