#!/usr/bin/env php
<?php

declare(strict_types=1);

require __DIR__ . '/../vendor/autoload.php';

use App\Queue\JobQueue;

/**
 * Background worker untuk processing jobs
 * Usage: php bin/worker.php [queue_name]
 */

$queueName = $argv[1] ?? 'default';

echo "[Worker] Starting worker for queue: {$queueName}\n";

// Setup Redis connection
$redis = new Redis();
$redis->connect('127.0.0.1', 6379);

$queue = new JobQueue($redis);

// Graceful shutdown handler
$shutdown = false;
pcntl_signal(SIGTERM, function() use (&$shutdown) {
    $shutdown = true;
    echo "[Worker] Shutdown signal received\n";
});
pcntl_signal(SIGINT, function() use (&$shutdown) {
    $shutdown = true;
    echo "[Worker] Interrupt signal received\n";
});

// Main loop
while (!$shutdown) {
    // Allow signal processing
    pcntl_signal_dispatch();

    $job = $queue->pop($queueName, 5);

    if ($job === null) {
        continue;
    }

    $jobId = $job['id'] ?? 'unknown';
    $payload = $job['payload'] ?? [];

    echo "[Worker] Processing job: {$jobId}\n";

    try {
        // Process job based on type
        processJob($payload);

        // Mark as completed
        $queue->ack($jobId);

        echo "[Worker] ✓ Job completed: {$jobId}\n";
    } catch (\Throwable $e) {
        echo "[Worker] ✗ Job failed: {$jobId} - {$e->getMessage()}\n";

        // Retry or move to failed queue
        $maxRetries = 3;
        if (($job['attempts'] ?? 0) < $maxRetries) {
            $queue->retry($queueName, $job, $maxRetries);
            echo "[Worker] → Job queued for retry\n";
        } else {
            $queue->fail($queueName, $job, $e->getMessage());
            echo "[Worker] → Job moved to failed queue\n";
        }
    }
}

echo "[Worker] Worker stopped gracefully\n";

/**
 * Process job based on type
 *
 * @param array<string, mixed> $payload
 */
function processJob(array $payload): void
{
    $type = $payload['type'] ?? null;

    match($type) {
        'email' => sendEmail($payload),
        'sms' => sendSms($payload),
        'notification' => sendNotification($payload),
        'export' => exportData($payload),
        default => throw new \InvalidArgumentException("Unknown job type: {$type}"),
    };
}

/**
 * @param array<string, mixed> $payload
 */
function sendEmail(array $payload): void
{
    $to = $payload['to'] ?? throw new \InvalidArgumentException('Missing email recipient');
    $subject = $payload['subject'] ?? 'No Subject';
    $body = $payload['body'] ?? '';

    // Simulate email sending
    sleep(1);

    echo "[Email] Sent to {$to}: {$subject}\n";
}

/**
 * @param array<string, mixed> $payload
 */
function sendSms(array $payload): void
{
    $phone = $payload['phone'] ?? throw new \InvalidArgumentException('Missing phone number');
    $message = $payload['message'] ?? '';

    // Simulate SMS sending
    sleep(1);

    echo "[SMS] Sent to {$phone}\n";
}

/**
 * @param array<string, mixed> $payload
 */
function sendNotification(array $payload): void
{
    $userId = $payload['user_id'] ?? throw new \InvalidArgumentException('Missing user_id');
    $message = $payload['message'] ?? '';

    // Simulate notification
    sleep(1);

    echo "[Notification] Sent to user {$userId}\n";
}

/**
 * @param array<string, mixed> $payload
 */
function exportData(array $payload): void
{
    $format = $payload['format'] ?? 'csv';

    // Simulate export
    sleep(2);

    echo "[Export] Generated {$format} export\n";
}
