<?php

declare(strict_types=1);

/**
 * OG Shortlink Production Installer
 * For cPanel Shared Hosting
 */
final class Installer
{
    private const MIN_PHP_VERSION = '8.1.0';
    private const REQUIRED_EXTENSIONS = [
        'pdo',
        'pdo_mysql',
        'curl',
        'json',
        'mbstring',
        'openssl',
        'fileinfo',
    ];
    private const OPTIONAL_EXTENSIONS = [
        'apcu' => 'Recommended for caching (performance)',
        'redis' => 'Optional for distributed caching',
    ];
    private const WRITABLE_DIRS = [
        'data',
        'data/cache',
        'data/logs',
        'config',
    ];

    private string $basePath;
    private array $errors = [];
    private array $warnings = [];

    public function __construct()
    {
        $this->basePath = dirname(__DIR__);
    }

    /**
     * Check all system requirements
     *
     * @return array{passed: bool, checks: array<string, array{status: string, message: string}>}
     */
    public function checkRequirements(): array
    {
        $checks = [];

        // PHP Version
        $phpVersion = PHP_VERSION;
        $phpPassed = version_compare($phpVersion, self::MIN_PHP_VERSION, '>=');
        $checks['php_version'] = [
            'status' => $phpPassed ? 'pass' : 'fail',
            'message' => "PHP {$phpVersion} " . ($phpPassed ? "(>= " . self::MIN_PHP_VERSION . ")" : "(requires >= " . self::MIN_PHP_VERSION . ")"),
        ];

        // Required Extensions
        foreach (self::REQUIRED_EXTENSIONS as $ext) {
            $loaded = extension_loaded($ext);
            $checks["ext_{$ext}"] = [
                'status' => $loaded ? 'pass' : 'fail',
                'message' => "Extension: {$ext}",
            ];
        }

        // Optional Extensions
        foreach (self::OPTIONAL_EXTENSIONS as $ext => $desc) {
            $loaded = extension_loaded($ext);
            $checks["ext_{$ext}"] = [
                'status' => $loaded ? 'pass' : 'warn',
                'message' => "Extension: {$ext} - {$desc}",
            ];
        }

        // Writable Directories
        foreach (self::WRITABLE_DIRS as $dir) {
            $fullPath = $this->basePath . '/' . $dir;
            $exists = is_dir($fullPath);
            $writable = $exists && is_writable($fullPath);

            if (!$exists) {
                // Try to create
                $created = @mkdir($fullPath, 0755, true);
                $writable = $created;
            }

            $checks["dir_{$dir}"] = [
                'status' => $writable ? 'pass' : 'fail',
                'message' => "Directory: {$dir} " . ($writable ? "(writable)" : "(not writable)"),
            ];
        }

        // Check .htaccess for Apache
        $htaccessExists = file_exists($this->basePath . '/public/.htaccess');
        $checks['htaccess'] = [
            'status' => $htaccessExists ? 'pass' : 'warn',
            'message' => '.htaccess file ' . ($htaccessExists ? 'exists' : 'missing - may need manual setup'),
        ];

        // Check mod_rewrite (if Apache)
        $modRewrite = function_exists('apache_get_modules') && in_array('mod_rewrite', apache_get_modules());
        $checks['mod_rewrite'] = [
            'status' => $modRewrite ? 'pass' : 'warn',
            'message' => 'mod_rewrite ' . ($modRewrite ? 'enabled' : 'status unknown (check manually)'),
        ];

        // Overall pass check (ignoring warnings)
        $passed = true;
        foreach ($checks as $check) {
            if ($check['status'] === 'fail') {
                $passed = false;
                break;
            }
        }

        return ['passed' => $passed, 'checks' => $checks];
    }

    /**
     * Test database connection
     *
     * @return array{success: bool, message: string, pdo?: \PDO}
     */
    public function testDatabase(string $host, string $name, string $user, string $pass, int $port = 3306): array
    {
        try {
            $dsn = "mysql:host={$host};port={$port};dbname={$name};charset=utf8mb4";
            $pdo = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);

            // Test query
            $pdo->query('SELECT 1');

            return ['success' => true, 'message' => 'Database connection successful', 'pdo' => $pdo];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Connection failed: ' . $e->getMessage()];
        }
    }

    /**
     * Install database schema
     */
    public function installDatabase(PDO $pdo): array
    {
        $results = [];

        try {
            // Read schema file
            $schemaFile = $this->basePath . '/database/schema.sql';
            if (!file_exists($schemaFile)) {
                return ['success' => false, 'message' => 'Schema file not found'];
            }

            $schema = file_get_contents($schemaFile);

            // Clean up schema for execution
            // Remove MariaDB-specific headers and comments
            $schema = preg_replace('/\/\*M!.*?\*\//', '', $schema);
            $schema = preg_replace('/\/\*![\d]+ (.*?) \*\//', '$1', $schema);

            // Split into individual statements
            $statements = $this->splitSqlStatements($schema);

            $pdo->beginTransaction();

            foreach ($statements as $stmt) {
                $stmt = trim($stmt);
                if ($stmt === '' || strpos($stmt, '--') === 0) {
                    continue;
                }

                // Skip certain statements that might fail on shared hosting
                if (stripos($stmt, 'LOCK TABLES') !== false ||
                    stripos($stmt, 'UNLOCK TABLES') !== false ||
                    stripos($stmt, 'SET @') !== false ||
                    stripos($stmt, 'SET TIME_ZONE') !== false ||
                    stripos($stmt, 'SET SQL_MODE') !== false ||
                    stripos($stmt, 'SET NAMES') !== false) {
                    continue;
                }

                try {
                    $pdo->exec($stmt);
                    $results[] = ['success' => true, 'statement' => substr($stmt, 0, 50) . '...'];
                } catch (PDOException $e) {
                    // Skip duplicate errors for DROP TABLE IF EXISTS
                    if (stripos($stmt, 'DROP TABLE') === false) {
                        $results[] = ['success' => false, 'statement' => substr($stmt, 0, 50) . '...', 'error' => $e->getMessage()];
                    }
                }
            }

            $pdo->commit();

            return ['success' => true, 'message' => 'Database schema installed', 'details' => $results];
        } catch (Throwable $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            return ['success' => false, 'message' => 'Installation failed: ' . $e->getMessage()];
        }
    }

    /**
     * Create admin user
     */
    public function createAdminUser(PDO $pdo, string $username, string $password, string $name): array
    {
        try {
            // Validate password
            if (strlen($password) < 8) {
                return ['success' => false, 'message' => 'Password must be at least 8 characters'];
            }

            // Check if user exists
            $stmt = $pdo->prepare('SELECT id FROM users WHERE username = ?');
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                return ['success' => false, 'message' => 'Username already exists'];
            }

            // Create user
            $hash = password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
            $stmt = $pdo->prepare('
                INSERT INTO users (username, password_hash, name, role, is_active)
                VALUES (?, ?, ?, ?, 1)
            ');
            $stmt->execute([$username, $hash, $name, 'admin']);

            return ['success' => true, 'message' => 'Admin user created successfully'];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Failed to create user: ' . $e->getMessage()];
        }
    }

    /**
     * Generate .env file
     */
    public function generateEnvFile(array $config): array
    {
        $envContent = <<<ENV
# OG Shortlink Configuration
# Generated by installer on {$config['_generated_at']}

# =============================================================================
# DATABASE
# =============================================================================
DB_DSN="mysql:host={$config['db_host']};port={$config['db_port']};dbname={$config['db_name']};charset=utf8mb4"
DB_USER="{$config['db_user']}"
DB_PASS="{$config['db_pass']}"

# =============================================================================
# APPLICATION
# =============================================================================
APP_ENV="production"
APP_DEBUG="false"
PUBLIC_BASE_URL="{$config['base_url']}"

# =============================================================================
# SECURITY
# =============================================================================
# Generate new keys: php -r "echo bin2hex(random_bytes(32));"
SESSION_SECRET="{$config['session_secret']}"
ENCRYPTION_KEY="{$config['encryption_key']}"

# =============================================================================
# SHORTLINK SETTINGS
# =============================================================================
SHORTLINK_API_KEY="{$config['api_key']}"
SHORTLINK_HTTPS_ONLY="true"
SHORTLINK_BLOCK_IP_LITERAL="true"
SHORTLINK_BLOCK_USERINFO="true"

# URL Signing (optional - enable for extra security)
URL_SIGN_ENABLED="false"
URL_SIGN_SECRET="{$config['url_sign_secret']}"

# =============================================================================
# TRAFFIC LOGGING
# =============================================================================
TRAFFIC_LOG_ENABLED="true"
TRAFFIC_LOG_ASYNC="true"

# =============================================================================
# IP INTELLIGENCE (Optional)
# =============================================================================
IP_INTEL_ENABLED="false"
# IP_INTEL_PROVIDER="ipinfo"
# IPINFO_TOKEN=""

# =============================================================================
# SHIM LINK (Optional)
# =============================================================================
SHIM_ENABLED="false"
# SHIM_BASE_URL=""
# SHIM_SECRET=""

ENV;

        $envFile = $this->basePath . '/.env';

        // Backup existing .env if exists
        if (file_exists($envFile)) {
            $backupFile = $this->basePath . '/.env.backup.' . date('YmdHis');
            copy($envFile, $backupFile);
        }

        if (file_put_contents($envFile, $envContent) === false) {
            return ['success' => false, 'message' => 'Failed to write .env file'];
        }

        // Secure the file
        chmod($envFile, 0600);

        return ['success' => true, 'message' => '.env file created successfully'];
    }

    /**
     * Generate random secure keys
     */
    public function generateSecureKeys(): array
    {
        return [
            'session_secret' => bin2hex(random_bytes(32)),
            'encryption_key' => bin2hex(random_bytes(32)),
            'api_key' => bin2hex(random_bytes(24)),
            'url_sign_secret' => bin2hex(random_bytes(32)),
        ];
    }

    /**
     * Set proper file permissions
     */
    public function setPermissions(): array
    {
        $results = [];

        // Directories that need to be writable
        $writableDirs = [
            'data' => 0755,
            'data/cache' => 0755,
            'data/logs' => 0755,
            'config' => 0755,
        ];

        foreach ($writableDirs as $dir => $perms) {
            $path = $this->basePath . '/' . $dir;
            if (is_dir($path)) {
                if (@chmod($path, $perms)) {
                    $results[] = ['success' => true, 'path' => $dir, 'perms' => decoct($perms)];
                } else {
                    $results[] = ['success' => false, 'path' => $dir, 'message' => 'Failed to set permissions'];
                }
            }
        }

        // Secure sensitive files
        $secureFiles = [
            '.env' => 0600,
            'config/facebook_protection.php' => 0644,
        ];

        foreach ($secureFiles as $file => $perms) {
            $path = $this->basePath . '/' . $file;
            if (file_exists($path)) {
                @chmod($path, $perms);
            }
        }

        return ['success' => true, 'details' => $results];
    }

    /**
     * Get cron job instructions
     */
    public function getCronInstructions(): string
    {
        $phpPath = PHP_BINDIR . '/php';
        $basePath = $this->basePath;

        return <<<CRON
# =============================================================================
# CRON JOBS FOR OG SHORTLINK
# =============================================================================
# Add these to your cPanel Cron Jobs (every minute recommended):

# Traffic Queue Worker (processes async traffic logs)
* * * * * {$phpPath} {$basePath}/bin/traffic-worker.php >> {$basePath}/data/logs/worker.log 2>&1

# Optional: Daily log rotation
0 0 * * * find {$basePath}/data/logs -name "*.log" -mtime +30 -delete

CRON;
    }

    /**
     * Remove installer after successful installation
     */
    public function selfDestruct(): array
    {
        $installDir = __DIR__;

        // Create marker file to prevent re-installation
        $lockFile = $this->basePath . '/data/.installed';
        file_put_contents($lockFile, date('c') . "\n" . 'Installed by: ' . ($_SERVER['REMOTE_ADDR'] ?? 'CLI'));

        // Delete installer files
        $files = [
            $installDir . '/index.php',
            $installDir . '/Installer.php',
            $installDir . '/templates/layout.php',
            $installDir . '/templates/step1.php',
            $installDir . '/templates/step2.php',
            $installDir . '/templates/step3.php',
            $installDir . '/templates/step4.php',
            $installDir . '/templates/complete.php',
            $installDir . '/assets/installer.css',
        ];

        $deleted = [];
        foreach ($files as $file) {
            if (file_exists($file)) {
                if (@unlink($file)) {
                    $deleted[] = basename($file);
                }
            }
        }

        // Try to remove directories
        @rmdir($installDir . '/templates');
        @rmdir($installDir . '/assets');
        @rmdir($installDir);

        return ['success' => true, 'deleted' => $deleted];
    }

    /**
     * Check if already installed
     */
    public function isInstalled(): bool
    {
        return file_exists($this->basePath . '/data/.installed') ||
               file_exists($this->basePath . '/.env');
    }

    /**
     * Split SQL into individual statements
     */
    private function splitSqlStatements(string $sql): array
    {
        // Remove comments
        $sql = preg_replace('/--.*$/m', '', $sql);
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);

        // Split by semicolons (but not inside strings)
        $statements = [];
        $current = '';
        $inString = false;
        $stringChar = '';

        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];

            if (!$inString && ($char === '"' || $char === "'")) {
                $inString = true;
                $stringChar = $char;
            } elseif ($inString && $char === $stringChar && ($i === 0 || $sql[$i - 1] !== '\\')) {
                $inString = false;
            }

            if ($char === ';' && !$inString) {
                $statements[] = trim($current);
                $current = '';
            } else {
                $current .= $char;
            }
        }

        if (trim($current) !== '') {
            $statements[] = trim($current);
        }

        return $statements;
    }

    /**
     * Detect base URL
     */
    public function detectBaseUrl(): string
    {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';

        // Remove /install from path
        $path = dirname($_SERVER['SCRIPT_NAME']);
        $path = str_replace('/install', '', $path);
        $path = rtrim($path, '/');

        return "{$protocol}://{$host}{$path}";
    }
}
