<?php

declare(strict_types=1);

/**
 * OG Shortlink Production Installer
 * For cPanel Shared Hosting
 *
 * Usage:
 * 1. Upload all files to your hosting
 * 2. Navigate to https://yourdomain.com/install/
 * 3. Follow the installation wizard
 * 4. Delete the install folder after completion
 */

// Error handling - show errors during installation
error_reporting(E_ALL);
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
ini_set('log_errors', '1');

// Check PHP version first
if (version_compare(PHP_VERSION, '8.1.0', '<')) {
    die('PHP 8.1 or higher is required. Current version: ' . PHP_VERSION);
}

// Start session for multi-step form
session_start();

// Load installer class
require_once __DIR__ . '/Installer.php';

$installer = new Installer();

// Check if already installed
if ($installer->isInstalled() && !isset($_GET['force'])) {
    http_response_code(403);
    die('<!DOCTYPE html><html><head><title>Already Installed</title></head><body style="font-family:sans-serif;text-align:center;padding:50px;">
    <h1>Already Installed</h1>
    <p>OG Shortlink is already installed. For security, the installer has been disabled.</p>
    <p><a href="../admin/">Go to Admin Panel</a></p>
    </body></html>');
}

// Determine current step
$step = (int) ($_GET['step'] ?? $_POST['step'] ?? 1);
$step = max(1, min(5, $step));

$error = null;
$content = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Test database connection (AJAX)
    if ($action === 'test_db') {
        header('Content-Type: application/json');
        $result = $installer->testDatabase(
            $_POST['db_host'] ?? 'localhost',
            $_POST['db_name'] ?? '',
            $_POST['db_user'] ?? '',
            $_POST['db_pass'] ?? '',
            (int) ($_POST['db_port'] ?? 3306)
        );
        echo json_encode(['success' => $result['success'], 'message' => $result['message']]);
        exit;
    }

    // Process step submissions
    if ($action === 'next') {
        switch ($step) {
            case 1:
                // Requirements check passed, go to step 2
                $step = 2;
                break;

            case 2:
                // Install database
                $dbResult = $installer->testDatabase(
                    $_POST['db_host'] ?? 'localhost',
                    $_POST['db_name'] ?? '',
                    $_POST['db_user'] ?? '',
                    $_POST['db_pass'] ?? '',
                    (int) ($_POST['db_port'] ?? 3306)
                );

                if (!$dbResult['success']) {
                    $error = $dbResult['message'];
                } else {
                    // Install schema
                    $installResult = $installer->installDatabase($dbResult['pdo']);
                    if (!$installResult['success']) {
                        $error = $installResult['message'];
                    } else {
                        // Store DB config in session
                        $_SESSION['db_host'] = $_POST['db_host'];
                        $_SESSION['db_port'] = $_POST['db_port'];
                        $_SESSION['db_name'] = $_POST['db_name'];
                        $_SESSION['db_user'] = $_POST['db_user'];
                        $_SESSION['db_pass'] = $_POST['db_pass'];
                        $step = 3;
                    }
                }
                break;

            case 3:
                // Generate .env file
                $config = [
                    'db_host' => $_POST['db_host'] ?? $_SESSION['db_host'] ?? '',
                    'db_port' => $_POST['db_port'] ?? $_SESSION['db_port'] ?? '3306',
                    'db_name' => $_POST['db_name'] ?? $_SESSION['db_name'] ?? '',
                    'db_user' => $_POST['db_user'] ?? $_SESSION['db_user'] ?? '',
                    'db_pass' => $_POST['db_pass'] ?? $_SESSION['db_pass'] ?? '',
                    'base_url' => rtrim($_POST['base_url'] ?? '', '/'),
                    'session_secret' => $_POST['session_secret'] ?? '',
                    'encryption_key' => $_POST['encryption_key'] ?? '',
                    'api_key' => $_POST['api_key'] ?? '',
                    'url_sign_secret' => $_POST['url_sign_secret'] ?? '',
                    '_generated_at' => date('Y-m-d H:i:s'),
                ];

                $envResult = $installer->generateEnvFile($config);
                if (!$envResult['success']) {
                    $error = $envResult['message'];
                } else {
                    $_SESSION['base_url'] = $config['base_url'];
                    $step = 4;
                }
                break;

            case 4:
                // Create admin user
                $password = $_POST['admin_password'] ?? '';
                $confirm = $_POST['admin_password_confirm'] ?? '';

                if ($password !== $confirm) {
                    $error = 'Passwords do not match';
                } elseif (strlen($password) < 8) {
                    $error = 'Password must be at least 8 characters';
                } else {
                    // Connect to database
                    $dbResult = $installer->testDatabase(
                        $_SESSION['db_host'],
                        $_SESSION['db_name'],
                        $_SESSION['db_user'],
                        $_SESSION['db_pass'],
                        (int) $_SESSION['db_port']
                    );

                    if (!$dbResult['success']) {
                        $error = 'Database connection failed: ' . $dbResult['message'];
                    } else {
                        $userResult = $installer->createAdminUser(
                            $dbResult['pdo'],
                            $_POST['admin_username'] ?? 'admin',
                            $password,
                            $_POST['admin_name'] ?? 'Administrator'
                        );

                        if (!$userResult['success']) {
                            $error = $userResult['message'];
                        } else {
                            // Set permissions
                            $installer->setPermissions();

                            // Self destruct
                            $installer->selfDestruct();

                            $step = 5;
                        }
                    }
                }
                break;
        }
    }
}

// Render current step
ob_start();

switch ($step) {
    case 1:
        $requirements = $installer->checkRequirements();
        include __DIR__ . '/templates/step1.php';
        break;

    case 2:
        include __DIR__ . '/templates/step2.php';
        break;

    case 3:
        $keys = $installer->generateSecureKeys();
        $baseUrl = $installer->detectBaseUrl();
        include __DIR__ . '/templates/step3.php';
        break;

    case 4:
        include __DIR__ . '/templates/step4.php';
        break;

    case 5:
        $baseUrl = $_SESSION['base_url'] ?? $installer->detectBaseUrl();
        $cronInstructions = $installer->getCronInstructions();
        include __DIR__ . '/templates/complete.php';

        // Clear session
        session_destroy();
        break;
}

$content = ob_get_clean();
$currentStep = $step;

// Render layout
include __DIR__ . '/templates/layout.php';
