<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Db;
use App\Env;
use App\Http;
use App\ShortlinkRepository;

require_once __DIR__ . '/../../vendor/autoload.php';
Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

$csrfToken = Auth::generateCsrfToken();

// Get and validate shortlink code from query param
$code = Http::validateCode((string) ($_GET['code'] ?? ''));
if ($code === '') {
    header('Location: /admin/index.php');
    exit;
}

// Validate shortlink exists
$pdo = Db::pdo();
$repo = new ShortlinkRepository($pdo);
$shortlink = $repo->findByCode($code);

if ($shortlink === null) {
    header('Location: /admin/index.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Analytics - <?= e($code) ?> | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Analytics</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php" aria-current="page">Analytics</a>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">

        <div class="topbar">
          <div>
            <h1 class="page-title">Traffic Analytics <span class="tag"><?= e($code) ?></span></h1>
            <div class="page-sub">Monitor shortlink performance and visitor insights</div>
          </div>
          <div class="actions">
            <button class="btn btn-sm time-filter-btn active" data-days="7">7 Days</button>
            <button class="btn btn-sm time-filter-btn" data-days="30">30 Days</button>
            <button class="btn btn-sm time-filter-btn" data-days="90">90 Days</button>
          </div>
        </div>

        <div class="grid" id="statsGrid">
          <div class="card kpi" style="grid-column: span 12; text-align: center;">
            <span class="muted">Loading statistics...</span>
          </div>
        </div>

        <div class="grid">
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">Visits Over Time</h3>
                <div class="card-sub">Daily traffic trend</div>
              </div>
            </div>
            <canvas id="visitsChart" style="height: 300px;"></canvas>
          </div>
        </div>

        <div class="grid" id="topLists" style="grid-template-columns: repeat(3, 1fr);">
          <div class="card" style="text-align: center;">
            <span class="muted">Loading top lists...</span>
          </div>
        </div>

        <div class="grid">
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">Recent Visitors</h3>
                <div class="card-sub">Latest visitor activity</div>
              </div>
            </div>
            <div class="table-wrap">
              <table>
                <thead>
                  <tr>
                    <th>Time</th>
                    <th>IP Address</th>
                    <th>Location</th>
                    <th>Device</th>
                    <th>Browser</th>
                    <th>OS</th>
                    <th>Risk</th>
                  </tr>
                </thead>
                <tbody id="visitorsTable">
                  <tr>
                    <td colspan="7" style="text-align: center;"><span class="muted">Loading visitors...</span></td>
                  </tr>
                </tbody>
              </table>
            </div>
            <div id="pagination" style="display: flex; justify-content: center; gap: 8px; margin-top: 10px;"></div>
          </div>
        </div>

        <div class="footer">
          <p>&copy; <?= date('Y') ?> OG Shortlink</p>
        </div>
      </div>
    </main>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
  <script src="/admin/assets/toast.js" nonce="<?= e($nonce) ?>"></script>
  <script nonce="<?= e($nonce) ?>">
  (function(){
    'use strict';

    const csrfToken = <?= json_encode($csrfToken, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) ?>;
    const shortlinkCode = <?= json_encode($code, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) ?>;

    let currentDays = 7;
    let currentPage = 1;
    let chart = null;

    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
      loadAnalytics();
      setupTimeFilters();
    });

    function setupTimeFilters() {
      document.querySelectorAll('.time-filter-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
          currentDays = parseInt(btn.dataset.days);

          document.querySelectorAll('.time-filter-btn').forEach(function(b) {
            b.classList.remove('active');
          });
          btn.classList.add('active');

          loadAnalytics();
        });
      });
    }

    async function loadAnalytics() {
      await Promise.all([
        loadSummary(),
        loadChart(),
        loadVisitors()
      ]);
    }

    async function loadSummary() {
      try {
        const response = await fetch('/admin/api.php?action=analytics_summary&code=' + encodeURIComponent(shortlinkCode) + '&days=' + currentDays);
        const data = await response.json();

        if (!data.ok) {
          showToast('Failed to load summary', 'error');
          return;
        }

        const summary = data.summary;

        document.getElementById('statsGrid').innerHTML =
          '<div class="card kpi"><div class="kpi-label"><span>Total Visits</span><span class="dot"></span></div>' +
          '<div class="kpi-value">' + summary.total_visits.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">All traffic</span></div></div>' +

          '<div class="card kpi"><div class="kpi-label"><span>Unique IPs</span><span class="dot"></span></div>' +
          '<div class="kpi-value">' + summary.unique_ips.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">Distinct visitors</span></div></div>' +

          '<div class="card kpi"><div class="kpi-label"><span>Mobile</span><span class="dot"></span></div>' +
          '<div class="kpi-value">' + summary.mobile_visits.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">Mobile devices</span></div></div>' +

          '<div class="card kpi"><div class="kpi-label"><span>Desktop</span><span class="dot"></span></div>' +
          '<div class="kpi-value">' + summary.desktop_visits.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">Desktop devices</span></div></div>' +

          '<div class="card kpi"><div class="kpi-label"><span>Bots</span><span class="dot dot-warn"></span></div>' +
          '<div class="kpi-value">' + summary.bot_visits.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">Automated</span></div></div>' +

          '<div class="card kpi"><div class="kpi-label"><span>Risky</span><span class="dot dot-bad"></span></div>' +
          '<div class="kpi-value">' + summary.risky_visits.toLocaleString() + '</div>' +
          '<div class="kpi-meta"><span class="muted">High risk</span></div></div>';

        renderTopLists(summary);
      } catch (error) {
        console.error('Failed to load summary:', error);
        showToast('Failed to load summary', 'error');
      }
    }

    function renderTopLists(summary) {
      var topCountriesHtml = Object.entries(summary.top_countries || {})
        .map(function(entry) {
          return '<div style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid var(--border-color);">' +
            '<span>' + (entry[0] || 'Unknown') + '</span>' +
            '<span class="muted">' + entry[1].toLocaleString() + '</span></div>';
        }).join('') || '<div style="text-align: center; padding: 12px;"><span class="muted">No data</span></div>';

      var topDevicesHtml = Object.entries(summary.top_devices || {})
        .map(function(entry) {
          return '<div style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid var(--border-color);">' +
            '<span>' + (entry[0] || 'Unknown') + '</span>' +
            '<span class="muted">' + entry[1].toLocaleString() + '</span></div>';
        }).join('') || '<div style="text-align: center; padding: 12px;"><span class="muted">No data</span></div>';

      var topBrowsersHtml = Object.entries(summary.top_browsers || {})
        .map(function(entry) {
          return '<div style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid var(--border-color);">' +
            '<span>' + (entry[0] || 'Unknown') + '</span>' +
            '<span class="muted">' + entry[1].toLocaleString() + '</span></div>';
        }).join('') || '<div style="text-align: center; padding: 12px;"><span class="muted">No data</span></div>';

      document.getElementById('topLists').innerHTML =
        '<div class="card"><div class="card-head"><h3 class="card-title">Top Countries</h3></div>' + topCountriesHtml + '</div>' +
        '<div class="card"><div class="card-head"><h3 class="card-title">Top Devices</h3></div>' + topDevicesHtml + '</div>' +
        '<div class="card"><div class="card-head"><h3 class="card-title">Top Browsers</h3></div>' + topBrowsersHtml + '</div>';
    }

    async function loadChart() {
      try {
        const response = await fetch('/admin/api.php?action=analytics_chart&code=' + encodeURIComponent(shortlinkCode) + '&days=' + currentDays);
        const data = await response.json();

        if (!data.ok) {
          showToast('Failed to load chart', 'error');
          return;
        }

        const chartData = data.chart;

        if (chart) {
          chart.destroy();
        }

        const ctx = document.getElementById('visitsChart').getContext('2d');
        chart = new Chart(ctx, {
          type: 'line',
          data: {
            labels: chartData.dates,
            datasets: [{
              label: 'Visits',
              data: chartData.visits,
              borderColor: 'hsl(240 5.9% 10%)',
              backgroundColor: 'rgba(15, 23, 42, 0.05)',
              tension: 0.4,
              fill: true
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
              legend: { display: false }
            },
            scales: {
              y: {
                beginAtZero: true,
                ticks: { precision: 0 }
              }
            }
          }
        });
      } catch (error) {
        console.error('Failed to load chart:', error);
        showToast('Failed to load chart', 'error');
      }
    }

    async function loadVisitors(page) {
      if (page === undefined) page = 1;
      currentPage = page;

      try {
        const response = await fetch('/admin/api.php?action=analytics_visitors&code=' + encodeURIComponent(shortlinkCode) + '&page=' + page + '&page_size=50');
        const data = await response.json();

        if (!data.ok) {
          showToast('Failed to load visitors', 'error');
          return;
        }

        const visitors = data.visitors || [];
        const total = data.total || 0;

        var visitorsHtml = visitors.map(function(v) {
          var location = [v.city, v.country_name].filter(Boolean).join(', ') || 'Unknown';
          var device = v.device_brand || v.device_type || 'Unknown';
          var riskBadge = v.is_bot ? '<span class="tag" style="background: #fef3c7; color: #92400e;">Bot</span>' :
            v.risk_score >= 75 ? '<span class="tag" style="background: #fee2e2; color: #991b1b;">Risky</span>' :
            '<span class="tag" style="background: #d1fae5; color: #065f46;">Clean</span>';

          return '<tr>' +
            '<td>' + new Date(v.visited_at).toLocaleString() + '</td>' +
            '<td>' + v.ip_address + '</td>' +
            '<td>' + location + '</td>' +
            '<td>' + device + '</td>' +
            '<td>' + (v.browser_name || 'Unknown') + '</td>' +
            '<td>' + (v.os_name || 'Unknown') + '</td>' +
            '<td>' + riskBadge + '</td>' +
            '</tr>';
        }).join('') || '<tr><td colspan="7" style="text-align: center;"><span class="muted">No visitors found</span></td></tr>';

        document.getElementById('visitorsTable').innerHTML = visitorsHtml;

        renderPagination(total, 50);
      } catch (error) {
        console.error('Failed to load visitors:', error);
        showToast('Failed to load visitors', 'error');
      }
    }

    function renderPagination(total, pageSize) {
      const totalPages = Math.ceil(total / pageSize);

      if (totalPages <= 1) {
        document.getElementById('pagination').innerHTML = '';
        return;
      }

      var paginationHtml =
        '<button class="btn btn-sm" ' + (currentPage <= 1 ? 'disabled' : '') + ' onclick="loadVisitors(' + (currentPage - 1) + ')">Previous</button>' +
        '<span style="padding: 0 8px;">Page ' + currentPage + ' of ' + totalPages + '</span>' +
        '<button class="btn btn-sm" ' + (currentPage >= totalPages ? 'disabled' : '') + ' onclick="loadVisitors(' + (currentPage + 1) + ')">Next</button>';

      document.getElementById('pagination').innerHTML = paginationHtml;
    }
  })();
  </script>
</body>
</html>
