# Global CSS Style Guide

**File**: `public/admin/assets/global.css`

**Purpose**: Modern, clean CSS framework untuk form-based pages dengan centered card layout.

---

## 🎨 Design System

### CSS Variables

```css
--ring-color: hsl(240 5.9% 10%)          /* Focus ring color */
--muted-foreground: hsl(240 3.8% 46.1%) /* Muted text color */
--border-color: hsl(240 5.9% 90%)       /* Border color */
--primary: hsl(240 5.9% 10%)            /* Primary button bg */
--primary-foreground: hsl(0 0% 98%)     /* Primary button text */
--page-bg: hsl(210 20% 97%)             /* Page background */
--card-bg: hsl(0 0% 100%)               /* Card background */
--danger: #dc2626                        /* Error/danger color */
--danger-soft: #fef2f2                   /* Error background */
--radius: 0.3rem                         /* Border radius */
```

---

## 📦 Components

### 1. Layout

**Body**:
- Centered flexbox layout
- Radial gradient background
- Full viewport height
- Responsive padding

**Wrap**:
- Max width: 420px
- Container for card

### 2. Card

**`.card`**:
- White background with shadow
- Rounded corners (0.3rem)
- Subtle border
- Padding: 18px 16px 14px

**`.card-header`**:
- Title section
- Bottom margin: 10px

### 3. Typography

**`.title`**:
- Font size: 1.1rem
- Font weight: 600
- Main heading

**`.subtitle`**:
- Font size: 0.85rem
- Muted color
- Subtitle/description

### 4. Form Elements

**`.field`**:
- Form field container
- Flexbox column layout
- Gap: 3px
- Bottom margin: 10px

**`label`**:
- Font size: 0.8rem
- Muted color

**`.label-required::after`**:
- Adds red asterisk (*) after label
- For required fields

**`input[type="text"], input[type="password"]`**:
- Rounded borders
- Light gray background (#f9fafb)
- Focus state with ring effect
- Smooth transitions

### 5. Buttons

**`.btn`**:
- Base button style
- Full width
- Flexbox centered
- Smooth transitions

**`.btn-primary`**:
- Dark background
- White text
- Hover effect with slight lift

### 6. Error Display

**`.error-box`**:
- Red background (soft)
- Red border
- Red text
- Small padding

### 7. Utilities

**`.footer`**:
- Small text (0.75rem)
- Muted color
- Centered

**`.badge-env`**:
- Small rounded badge
- For environment indicators

---

## 🚀 Usage

### Basic Page Template

```html
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Page Title</title>
    <link rel="stylesheet" href="/admin/assets/global.css">
</head>
<body>
    <div class="wrap">
        <div class="card">
            <div class="card-header">
                <div class="title">Title</div>
                <div class="subtitle">Subtitle</div>
            </div>

            <!-- Error box (optional) -->
            <div class="error-box">Error message</div>

            <!-- Form fields -->
            <div class="field">
                <label for="email" class="label-required">Email</label>
                <input type="text" id="email" name="email" required>
            </div>

            <div class="field">
                <label for="password" class="label-required">Password</label>
                <input type="password" id="password" name="password" required>
            </div>

            <!-- Button -->
            <button class="btn btn-primary" type="submit">
                Submit
            </button>

            <!-- Footer (optional) -->
            <div class="footer">
                Footer text
            </div>
        </div>
    </div>
</body>
</html>
```

---

## 📋 Component Examples

### Required Field

```html
<div class="field">
    <label for="username" class="label-required">Username</label>
    <input type="text" id="username" name="username" required>
</div>
```

### Optional Field

```html
<div class="field">
    <label for="bio">Bio</label>
    <input type="text" id="bio" name="bio">
</div>
```

### Error Message

```html
<?php if ($error): ?>
    <div class="error-box"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>
```

### Primary Button

```html
<button class="btn btn-primary" type="submit">
    Login
</button>
```

### Environment Badge

```html
<div class="footer">
    Version 1.0
    <span class="badge-env">Development</span>
</div>
```

---

## 🎯 Use Cases

### Perfect For:

✅ **Login pages**
✅ **Registration forms**
✅ **Password reset**
✅ **Simple form pages**
✅ **Single-card layouts**

### Not Suitable For:

❌ Dashboard layouts
❌ Multi-column layouts
❌ Data tables
❌ Complex admin panels

*(For dashboards, use `app.css` instead)*

---

## 🎨 Color Palette

| Color | Hex/HSL | Usage |
|-------|---------|-------|
| **Primary** | `hsl(240 5.9% 10%)` | Buttons, focus rings |
| **Foreground** | `hsl(0 0% 98%)` | Button text |
| **Muted** | `hsl(240 3.8% 46.1%)` | Labels, subtitles |
| **Border** | `hsl(240 5.9% 90%)` | Input borders, card borders |
| **Background** | `hsl(210 20% 97%)` | Page background |
| **Card** | `hsl(0 0% 100%)` | Card background (white) |
| **Danger** | `#dc2626` | Error text |
| **Danger Soft** | `#fef2f2` | Error background |

---

## 📱 Responsive Behavior

### Desktop (>480px):
- Card padding: 18px 16px 14px
- Max width: 420px
- Centered layout

### Mobile (≤480px):
- Card padding: 14px 12px 12px (reduced)
- Full width with 16px side margins
- Same centered layout

---

## 🔄 Migration from Old Style

### From `app.css` to `global.css`:

```diff
- <link rel="stylesheet" href="/admin/assets/app.css">
+ <link rel="stylesheet" href="/admin/assets/global.css">

- <main class="container">
-     <section class="card">
+ <div class="wrap">
+     <div class="card">

- <input class="input" type="text">
+ <input type="text">

- <button class="btn" type="submit">
+ <button class="btn btn-primary" type="submit">
```

---

## 🎭 Visual Examples

### Login Form:

```
┌─────────────────────────────┐
│  Login                      │
│  Enter your credentials     │
│                             │
│  Email *                    │
│  ┌───────────────────────┐ │
│  │                       │ │
│  └───────────────────────┘ │
│                             │
│  Password *                 │
│  ┌───────────────────────┐ │
│  │ ••••••••              │ │
│  └───────────────────────┘ │
│                             │
│  ┌───────────────────────┐ │
│  │       Login           │ │
│  └───────────────────────┘ │
│                             │
│  Version 1.0 [Dev]          │
└─────────────────────────────┘
```

---

## 🛠️ Customization

### Change Primary Color:

```css
:root {
    --primary: hsl(220 90% 50%);  /* Blue */
    --primary-foreground: #ffffff;
}
```

### Change Card Width:

```css
.wrap {
    max-width: 600px;  /* Wider */
}
```

### Add Dark Mode:

```css
@media (prefers-color-scheme: dark) {
    :root {
        --page-bg: hsl(240 10% 10%);
        --card-bg: hsl(240 10% 15%);
        --border-color: hsl(240 10% 25%);
        /* ... etc */
    }
}
```

---

## 📚 Related Files

- **Dashboard CSS**: `public/admin/assets/app.css`
- **Login Page**: `public/admin/login.php`
- **Global CSS**: `public/admin/assets/global.css` (this file)

---

## ✅ Checklist for New Pages

When creating a new page with global.css:

- [ ] Include `<link rel="stylesheet" href="/admin/assets/global.css">`
- [ ] Use `.wrap` as main container
- [ ] Use `.card` for content container
- [ ] Add `.card-header` with `.title` and `.subtitle`
- [ ] Use `.field` for form fields
- [ ] Add `.label-required` for required fields
- [ ] Use `.btn.btn-primary` for primary action
- [ ] Add `.error-box` for error messages (if needed)
- [ ] Test responsive layout on mobile

---

**Status**: ✅ Ready to use
**Location**: `public/admin/assets/global.css`
**Documentation**: This file
**Last Updated**: 2025-12-19
