/**
 * Theme System JavaScript
 * Handles dark/light theme toggle with localStorage persistence
 */
(function() {
  'use strict';

  const THEME_KEY = 'admin-theme';
  const THEME_LIGHT = 'light';
  const THEME_DARK = 'dark';

  /**
   * Theme Manager
   */
  const ThemeManager = {
    /**
     * Get current theme from localStorage or system preference
     */
    getTheme() {
      const stored = localStorage.getItem(THEME_KEY);
      if (stored) {
        return stored;
      }

      // Check system preference
      if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
        return THEME_DARK;
      }

      return THEME_LIGHT;
    },

    /**
     * Set theme
     */
    setTheme(theme) {
      const validTheme = theme === THEME_DARK ? THEME_DARK : THEME_LIGHT;

      // Add transition class
      document.body.classList.add('theme-transitioning');

      // Set theme attribute
      document.documentElement.setAttribute('data-theme', validTheme);

      // Save to localStorage
      localStorage.setItem(THEME_KEY, validTheme);

      // Remove transition class after animation
      setTimeout(() => {
        document.body.classList.remove('theme-transitioning');
      }, 500);

      // Dispatch custom event
      window.dispatchEvent(new CustomEvent('themechange', { detail: { theme: validTheme } }));

      return validTheme;
    },

    /**
     * Toggle theme
     */
    toggleTheme() {
      const current = this.getTheme();
      const newTheme = current === THEME_LIGHT ? THEME_DARK : THEME_LIGHT;
      return this.setTheme(newTheme);
    },

    /**
     * Initialize theme on page load
     */
    init() {
      const theme = this.getTheme();
      document.documentElement.setAttribute('data-theme', theme);

      // Listen for system theme changes
      if (window.matchMedia) {
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
          // Only auto-switch if user hasn't set a preference
          if (!localStorage.getItem(THEME_KEY)) {
            this.setTheme(e.matches ? THEME_DARK : THEME_LIGHT);
          }
        });
      }
    }
  };

  /**
   * Theme Toggle Button Component
   */
  const ThemeToggle = {
    /**
     * Create toggle button
     */
    create() {
      const button = document.createElement('button');
      button.className = 'theme-toggle';
      button.setAttribute('aria-label', 'Toggle theme');
      button.setAttribute('title', 'Toggle dark/light theme');
      button.innerHTML = `
        <svg class="theme-toggle-icon icon-moon" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z" />
        </svg>
        <svg class="theme-toggle-icon icon-sun" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z" />
        </svg>
      `;

      button.addEventListener('click', () => {
        ThemeManager.toggleTheme();
      });

      return button;
    },

    /**
     * Mount toggle button to page
     */
    mount() {
      const existing = document.querySelector('.theme-toggle');
      if (existing) {
        return;
      }

      const button = this.create();
      document.body.appendChild(button);
    }
  };

  /**
   * Initialize on DOM ready
   */
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      ThemeManager.init();
      ThemeToggle.mount();
    });
  } else {
    ThemeManager.init();
    ThemeToggle.mount();
  }

  /**
   * Expose to window for external access
   */
  window.ThemeManager = ThemeManager;

  /**
   * Add keyboard shortcut (Ctrl/Cmd + Shift + D)
   */
  document.addEventListener('keydown', (e) => {
    if ((e.ctrlKey || e.metaKey) && e.shiftKey && e.key === 'D') {
      e.preventDefault();
      ThemeManager.toggleTheme();
    }
  });

  /**
   * Debug helper (console)
   */
  if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
    console.log('[Theme] Current theme:', ThemeManager.getTheme());
    console.log('[Theme] Toggle with Ctrl/Cmd + Shift + D');
  }
})();
