/**
 * Toast Notification System
 * Usage: toast.success('Message'), toast.error('Error'), toast.info('Info')
 */
(function() {
  'use strict';

  const container = document.createElement('div');
  container.id = 'toast-container';
  container.style.cssText = 'position:fixed;top:20px;right:20px;z-index:99999;min-width:300px;max-width:400px;';
  document.body.appendChild(container);

  function show(message, type = 'info', duration = 4000) {
    const toast = document.createElement('div');
    toast.style.cssText = `
      background: #fff;
      border-left: 4px solid var(--toast-color, #333);
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      padding: 16px 20px;
      margin-bottom: 12px;
      border-radius: 4px;
      animation: slideIn 0.3s ease-out;
      cursor: pointer;
      transition: opacity 0.3s, transform 0.3s;
    `;

    const colors = {
      success: '#10b981',
      error: '#ef4444',
      warning: '#f59e0b',
      info: '#3b82f6'
    };

    toast.style.setProperty('--toast-color', colors[type] || colors.info);

    const icon = {
      success: '✓',
      error: '✗',
      warning: '!',
      info: 'ℹ'
    };

    toast.innerHTML = `
      <div style="display:flex;align-items:start;gap:12px;">
        <div style="
          width:24px;
          height:24px;
          border-radius:50%;
          background:var(--toast-color);
          color:#fff;
          display:flex;
          align-items:center;
          justify-content:center;
          font-weight:bold;
          font-size:14px;
          flex-shrink:0;
        ">${icon[type] || icon.info}</div>
        <div style="flex:1;padding-top:2px;">
          <div style="font-weight:600;margin-bottom:2px;text-transform:capitalize;">${type}</div>
          <div style="font-size:14px;color:#666;">${escapeHtml(message)}</div>
        </div>
      </div>
    `;

    toast.addEventListener('click', () => {
      toast.style.opacity = '0';
      toast.style.transform = 'translateX(400px)';
      setTimeout(() => toast.remove(), 300);
    });

    container.appendChild(toast);

    if (duration > 0) {
      setTimeout(() => {
        toast.style.opacity = '0';
        toast.style.transform = 'translateX(400px)';
        setTimeout(() => toast.remove(), 300);
      }, duration);
    }
  }

  function escapeHtml(str) {
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
  }

  // Add CSS animation
  const style = document.createElement('style');
  style.textContent = `
    @keyframes slideIn {
      from {
        transform: translateX(400px);
        opacity: 0;
      }
      to {
        transform: translateX(0);
        opacity: 1;
      }
    }
  `;
  document.head.appendChild(style);

  // Export to window
  window.toast = {
    success: (msg, duration) => show(msg, 'success', duration),
    error: (msg, duration) => show(msg, 'error', duration),
    warning: (msg, duration) => show(msg, 'warning', duration),
    info: (msg, duration) => show(msg, 'info', duration)
  };
})();
