<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Cache;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';
Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

// Get cache stats
$cacheDir = __DIR__ . '/../../data/cache';
$cacheFiles = [];
$totalSize = 0;

if (is_dir($cacheDir)) {
    $files = glob($cacheDir . '/*.cache');
    if ($files !== false) {
        foreach ($files as $file) {
            $size = filesize($file);
            if ($size !== false) {
                $totalSize += $size;
                $cacheFiles[] = [
                    'name' => basename($file),
                    'size' => $size,
                    'modified' => filemtime($file),
                ];
            }
        }
        usort($cacheFiles, fn($a, $b) => $b['modified'] - $a['modified']);
    }
}

$success = '';
$error = '';

// Handle clear cache action
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
    $csrf = (string) ($_POST['csrf_token'] ?? '');
    if (!Auth::verifyCsrfToken($csrf)) {
        $error = 'CSRF token invalid.';
    } else {
        $action = (string) ($_POST['action'] ?? '');
        if ($action === 'clear_all') {
            Cache::clear();
            $success = 'All cache cleared successfully!';
            header('Location: /admin/cache-stats.php');
            exit;
        }
    }
}

$csrf = Auth::ensureCsrfToken();

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Cache Statistics - OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Cache Stats</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/analytics.php">Analytics</a>
        <a href="/admin/cache-stats.php" aria-current="page">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">Cache Statistics</h1>
            <div class="page-sub">Monitor redirect cache performance</div>
          </div>
        </div>

        <?php if ($error !== '') : ?>
          <div class="error-box"><?= e($error) ?></div>
        <?php endif; ?>

        <?php if ($success !== '') : ?>
          <div class="success-box"><?= e($success) ?></div>
        <?php endif; ?>

        <div class="grid">
          <!-- Overview KPIs -->
          <div class="card kpi">
            <div class="kpi-label">
              <span>Cache Files</span>
              <span class="dot"></span>
            </div>
            <div class="kpi-value"><?= number_format(count($cacheFiles)) ?></div>
            <div class="kpi-meta">
              <span class="muted">Total entries</span>
            </div>
          </div>

          <div class="card kpi">
            <div class="kpi-label">
              <span>Total Size</span>
              <span class="dot"></span>
            </div>
            <div class="kpi-value"><?= number_format($totalSize / 1024, 2) ?></div>
            <div class="kpi-meta">
              <span class="muted">KB</span>
            </div>
          </div>

          <div class="card kpi">
            <div class="kpi-label">
              <span>Cache Type</span>
              <span class="dot"></span>
            </div>
            <div class="kpi-value">File</div>
            <div class="kpi-meta">
              <span class="muted">data/cache/</span>
            </div>
          </div>

          <div class="card kpi">
            <div class="kpi-label">
              <span>TTL</span>
              <span class="dot"></span>
            </div>
            <div class="kpi-value">5</div>
            <div class="kpi-meta">
              <span class="muted">minutes</span>
            </div>
          </div>

          <!-- Clear Cache -->
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">Cache Control</h3>
                <div class="card-sub">Clear all cached redirect data</div>
              </div>
            </div>

            <form id="clearCacheForm" method="post" action="/admin/cache-stats.php">
              <input type="hidden" name="csrf_token" value="<?= e($csrf) ?>">
              <input type="hidden" name="action" value="clear_all">
              <button class="btn btn-danger" type="submit">Clear All Cache</button>
            </form>
          </div>

          <!-- Cache Entries -->
          <?php if (!empty($cacheFiles)) : ?>
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">Recent Cache Entries</h3>
                <div class="card-sub">Latest <?= min(count($cacheFiles), 20) ?> cache files</div>
              </div>
            </div>

            <div class="table-wrap">
              <table>
                <thead>
                  <tr>
                    <th>File</th>
                    <th>Size</th>
                    <th>Last Modified</th>
                    <th>Age</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach (array_slice($cacheFiles, 0, 20) as $file) : ?>
                  <tr>
                    <td>
                      <span class="tag"><?= e($file['name']) ?></span>
                    </td>
                    <td><?= number_format($file['size']) ?> bytes</td>
                    <td class="muted"><?= date('Y-m-d H:i:s', $file['modified']) ?></td>
                    <td class="muted"><?= number_format(time() - $file['modified']) ?>s ago</td>
                  </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>

            <?php if (count($cacheFiles) > 20) : ?>
            <div class="muted" style="margin-top: 8px; font-size: 0.78rem;">
              Showing 20 of <?= count($cacheFiles) ?> total entries
            </div>
            <?php endif; ?>
          </div>
          <?php endif; ?>

          <!-- How Cache Works -->
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">How Cache Works</h3>
                <div class="card-sub">Understanding the caching mechanism</div>
              </div>
            </div>

            <div style="line-height: 1.6; font-size: 0.85rem;">
              <p><strong>Per-IP Caching:</strong> Setiap shortlink di-cache per IP address untuk mengurangi database queries.</p>

              <p><strong>Cache Key Format:</strong> <code>shortlink:{ip_hash}:{code}</code></p>

              <p><strong>Auto-Invalidation:</strong> Cache otomatis di-clear ketika:</p>
              <ul style="margin: 8px 0 12px 20px;">
                <li>Shortlink diupdate (target URL, title, dll)</li>
                <li>Shortlink di-enable/disable</li>
                <li>Redirect URL diupdate</li>
              </ul>

              <p><strong>Performance:</strong> Cache mengurangi database load untuk shortlink yang sering diakses.</p>

              <p><strong>Privacy:</strong> IP address di-hash dengan SHA-256 sebelum digunakan sebagai cache key.</p>
            </div>
          </div>
        </div>

        <div class="footer">
          <p>&copy; <?= date('Y') ?> OG Shortlink</p>
        </div>
      </div>
    </main>
  </div>

  <script nonce="<?= e($nonce) ?>">
  (function(){
    'use strict';

    // Auto-hide success message
    const successBox = document.querySelector('.success-box');
    if (successBox) {
      setTimeout(function(){
        successBox.style.transition = 'opacity 0.3s ease';
        successBox.style.opacity = '0';
        setTimeout(function(){ successBox.remove(); }, 300);
      }, 5000);
    }

    // Confirm before clearing cache
    const clearCacheForm = document.getElementById('clearCacheForm');
    if (clearCacheForm) {
      clearCacheForm.addEventListener('submit', function(e){
        if (!confirm('Clear all cache?\n\nThis will remove all cached redirect data.')) {
          e.preventDefault();
        }
      });
    }
  })();
  </script>
</body>
</html>
