<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';
\App\Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();
$csrf = Auth::ensureCsrfToken();
$userId = Auth::getUserId();
$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <title>My Domains | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">My Domains</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/users.php">Users</a>
        <?php endif; ?>
        <a href="/admin/domains.php" aria-current="page">Domains</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/global-domains.php">Global Domains</a>
        <?php endif; ?>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">My Domains</h1>
            <div class="page-sub">Manage your custom domains for shortlinks</div>
          </div>
        </div>

        <!-- DNS Setup Guide -->
        <div class="grid">
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">DNS Setup Guide</h3>
                <div class="card-sub">Configure your domain DNS records</div>
              </div>
              <div class="right">
                <button class="btn btn-sm" id="toggleDnsGuide" type="button">Show / Hide</button>
              </div>
            </div>

            <div id="dnsGuide" style="display: none; margin-top: 10px;">
              <div class="warning-box">
                <strong>How to configure your domain:</strong> Point your domain DNS records to this server to enable shortlinks.
              </div>

              <h4 style="font-size: 0.9rem; font-weight: 600; margin: 12px 0 8px;">1. DNS Record Types</h4>

              <div class="table-wrap">
                <table>
                  <thead>
                    <tr>
                      <th style="width:120px;">Record Type</th>
                      <th>Description</th>
                      <th style="width:200px;">Example</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr>
                      <td><span class="tag">A Record</span></td>
                      <td>Points domain directly to an IP address. Use this if you know the server IP.</td>
                      <td><code style="font-size:0.78rem;">192.0.2.1</code></td>
                    </tr>
                    <tr>
                      <td><span class="tag">CNAME</span></td>
                      <td>Points domain to another domain name (alias). Use this for CDN or load balancers.</td>
                      <td><code style="font-size:0.78rem;">server.example.com</code></td>
                    </tr>
                  </tbody>
                </table>
              </div>

              <h4 style="font-size: 0.9rem; font-weight: 600; margin: 12px 0 8px;">2. Get Server Information</h4>

              <div style="display: flex; gap: 12px; margin-bottom: 12px;">
                <div style="flex: 1;">
                  <div class="muted" style="margin-bottom: 4px; font-size: 0.78rem;">Server IP Address:</div>
                  <code id="serverIp" style="font-size: 0.82rem; background: #f9fafb; padding: 8px 12px; border-radius: var(--radius); display: block; border: 1px solid var(--border-color);">Loading...</code>
                </div>
                <div style="flex: 1;">
                  <div class="muted" style="margin-bottom: 4px; font-size: 0.78rem;">Server Hostname:</div>
                  <code id="serverHostname" style="font-size: 0.82rem; background: #f9fafb; padding: 8px 12px; border-radius: var(--radius); display: block; border: 1px solid var(--border-color);"><?= e($_SERVER['HTTP_HOST'] ?? 'N/A') ?></code>
                </div>
              </div>

              <h4 style="font-size: 0.9rem; font-weight: 600; margin: 12px 0 8px;">3. Configure DNS at Your Provider</h4>

              <div class="warning-box">
                <strong>Example Configuration:</strong>
                <ul style="margin: 8px 0 0 20px; padding: 0; font-size: 0.82rem;">
                  <li><strong>A Record:</strong> Host: <code style="font-size:0.78rem;">short.yourdomain.com</code> → Type: <code style="font-size:0.78rem;">A</code> → Value: <code id="exampleIp" style="font-size:0.78rem;">SERVER_IP</code></li>
                  <li><strong>CNAME:</strong> Host: <code style="font-size:0.78rem;">short.yourdomain.com</code> → Type: <code style="font-size:0.78rem;">CNAME</code> → Value: <code style="font-size:0.78rem;"><?= e($_SERVER['HTTP_HOST'] ?? 'server.example.com') ?></code></li>
                  <li><strong>Wildcard:</strong> Host: <code style="font-size:0.78rem;">*.yourdomain.com</code> → Type: <code style="font-size:0.78rem;">A</code> → Value: <code id="exampleIp2" style="font-size:0.78rem;">SERVER_IP</code></li>
                </ul>
              </div>

              <div class="muted" style="margin-top: 10px; font-size: 0.78rem;">
                DNS propagation can take 5 minutes to 48 hours. Use tools like <a href="https://dnschecker.org" target="_blank" rel="noopener" style="color: #0066cc;">dnschecker.org</a> to verify.
              </div>
            </div>
          </div>
        </div>

        <div class="grid">
          <!-- Form Card -->
          <div class="card">
            <?php if ($isAdmin) : ?>
            <div class="card-head">
              <h3 class="card-title" id="formTitle">Add New Domain</h3>
            </div>

            <form id="frm" autocomplete="off">
              <div>
                <label for="domain">Domain / Subdomain</label>
                <input type="text" id="domain" name="domain" placeholder="short.example.com" required>
                <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Enter your domain or subdomain untuk shortlink base URL</div>
              </div>

              <div style="margin-top: 12px;">
                <label for="provider">DNS Provider</label>
                <select id="provider" name="provider" required>
                  <option value="cloudflare" selected>Cloudflare DNS</option>
                  <option value="cpanel">cPanel</option>
                </select>

                <div class="warning-box" style="margin-top: 8px;">
                  <strong>ℹ️ Note:</strong> API credentials akan diambil dari <a href="/admin/settings.php" style="color: #0066cc; text-decoration: underline;">Security Configuration</a>. Pastikan sudah mengkonfigurasi Cloudflare atau cPanel API credentials di Settings.
                </div>

                <div id="cloudflareFields" style="display: block; margin-top: 12px;">
                  <label for="cf_zone_id">Zone ID (Optional)</label>
                  <input id="cf_zone_id" name="cf_zone_id" placeholder="Auto-detect if empty">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Leave empty to auto-detect from Cloudflare API</div>

                  <div style="margin-top: 12px;">
                    <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                      <input type="checkbox" id="cf_proxied" name="cf_proxied" value="1" style="width: 16px; height: 16px;">
                      <span style="font-size: 0.82rem;">Enable Cloudflare Proxy (Orange Cloud)</span>
                    </label>
                  </div>
                </div>

                <div id="cpanelFields" style="display: none; margin-top: 12px;">
                  <div class="muted" style="font-size: 0.82rem;">
                    cPanel credentials akan diambil dari Security Configuration.
                  </div>
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label for="target_type">DNS Record Type</label>
                <select id="target_type" name="target_type" required>
                  <option value="A" selected>A Record (IP Address)</option>
                  <option value="CNAME">CNAME Record (Domain)</option>
                </select>

                <div id="targetIpField" style="display: block; margin-top: 12px;">
                  <label for="target_ip">Target IP Address</label>
                  <input id="target_ip" name="target_ip" placeholder="192.0.2.1">
                </div>

                <div id="targetCnameField" style="display: none; margin-top: 12px;">
                  <label for="target_cname">Target CNAME</label>
                  <input id="target_cname" name="target_cname" placeholder="example.com">
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="is_wildcard" name="is_wildcard" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem; font-weight: 500;">Enable Wildcard DNS (*)</span>
                </label>
                <div class="muted" style="margin-top: 6px; font-size: 0.78rem;">
                  Allows subdomains like: code1.short.example.com, code2.short.example.com
                </div>
              </div>

              <div style="margin-top: 16px; display: flex; gap: 8px;">
                <button class="btn" id="submit" type="submit">Add Domain</button>
                <button class="btn btn-secondary" id="reset" type="button">Reset</button>
              </div>

              <div class="muted" style="margin-top: 12px; font-size: 0.78rem;">
                <strong>Note:</strong> Domain akan otomatis di-sync ke DNS provider saat di-save.
              </div>
            </form>

        <?php else : ?>
            <!-- User: Simplified form (basic options only, admin configures DNS) -->
            <div class="card-head">
              <h3 class="card-title" id="formTitleUser">Request Domain Configuration</h3>
              <div class="card-sub">Admin will configure the DNS settings</div>
            </div>

            <form id="frmUser" autocomplete="off">
              <div>
                <label for="user_domain">Domain / Subdomain</label>
                <input id="user_domain" name="user_domain" placeholder="short.example.com" readonly>
                <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Contact admin to add new domains</div>
              </div>

              <div style="margin-top: 12px;">
                <label for="user_record_type">DNS Record Type</label>
                <select id="user_record_type" name="user_record_type">
                  <option value="A" selected>A Record (IP Address)</option>
                  <option value="CNAME">CNAME (Domain Alias)</option>
                </select>
              </div>

              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="user_is_wildcard" name="user_is_wildcard" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem;">Enable Wildcard DNS (*)</span>
                </label>
                <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">
                  Wildcard allows all subdomains (*.yourdomain.com)
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="user_cf_proxied" name="user_cf_proxied" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem;">Enable Cloudflare Proxy (Orange Cloud)</span>
                </label>
                <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">
                  Enable Cloudflare CDN and protection features
                </div>
              </div>

              <div class="warning-box" style="margin-top: 16px;">
                <strong>Note:</strong> These are basic options. Admin configures DNS provider and API tokens.
              </div>
            </form>
            <?php endif; ?>
          </div>

          <!-- Table Card -->
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">My Domains</h3>
                <div class="card-sub" id="meta"></div>
              </div>
            </div>

            <div class="table-wrap">
              <table id="tbl">
                <thead>
                  <tr>
                    <th>Domain</th>
                    <th style="width:100px;">DNS Type</th>
                    <th style="width:150px;">Target</th>
                    <th style="width:100px;">Provider</th>
                    <th style="width:80px;">Wildcard</th>
                    <th style="width:100px;">Status</th>
                    <th style="width:180px;">Actions</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

<script nonce="<?= e($nonce) ?>">
(function(){
  'use strict';

  const csrf = document.querySelector('meta[name=\"csrf-token\"').getAttribute('content') || '';
  const isAdmin = <?= $isAdmin ? 'true' : 'false' ?>;
  let editingId = 0;

  const elTblBody = document.querySelector('#tbl tbody');
  const elMeta = document.getElementById('meta');
  const elFormTitle = document.getElementById('formTitle');
  const frm = document.getElementById('frm');

  // Provider switcher (admin only)
  if (isAdmin && document.getElementById('provider')) {
    document.getElementById('provider').addEventListener('change', function(){
      const provider = this.value;
      document.getElementById('cloudflareFields').style.display = provider === 'cloudflare' ? 'block' : 'none';
      document.getElementById('cpanelFields').style.display = provider === 'cpanel' ? 'block' : 'none';
    });
  }

  // Target type switcher (admin only)
  if (isAdmin && document.getElementById('target_type')) {
    document.getElementById('target_type').addEventListener('change', function(){
      const type = this.value;
      document.getElementById('targetIpField').style.display = type === 'A' ? 'block' : 'none';
      document.getElementById('targetCnameField').style.display = type === 'CNAME' ? 'block' : 'none';
    });
  }

  // Ensure toast is available
  if (typeof window.toast === 'undefined') {
    console.error('Toast library not loaded!');
    window.toast = {
      _show: function(msg, type){
        var toast = document.createElement('div');
        toast.style.cssText = 'position:fixed;top:20px;right:20px;background:#fff;border-left:4px solid ' +
          (type === 'error' ? '#ef4444' : type === 'success' ? '#10b981' : '#3b82f6') +
          ';box-shadow:0 4px 12px rgba(0,0,0,0.15);padding:16px 20px;border-radius:4px;z-index:99999;' +
          'max-width:400px;';
        toast.textContent = (type === 'error' ? '✗ ' : type === 'success' ? '✓ ' : 'ℹ ') + msg;
        document.body.appendChild(toast);
        setTimeout(function(){
          toast.style.transition = 'opacity 0.3s';
          toast.style.opacity = '0';
          setTimeout(function(){ toast.remove(); }, 300);
        }, 4000);
      },
      success: function(msg){ this._show(msg, 'success'); },
      error: function(msg){ this._show(msg, 'error'); },
      info: function(msg){ this._show(msg, 'info'); }
    };
  }

  function esc(s){
    return String(s)
      .replaceAll('&','&amp;')
      .replaceAll('<','&lt;')
      .replaceAll('>','&gt;')
      .replaceAll('"','&quot;')
      .replaceAll("'",'&#039;');
  }

  async function apiPost(action, payload){
    const res = await fetch('/admin/api.php?action=' + encodeURIComponent(action), {
      method: 'POST',
      credentials: 'same-origin',
      headers: {
        'Content-Type': 'application/json',
        'X-CSRF-Token': csrf
      },
      body: JSON.stringify(payload || {})
    });

    return await res.json();
  }

  async function apiGet(url){
    const res = await fetch(url, { credentials: 'same-origin' });
    return await res.json();
  }

  function setFormModeCreate(){
    editingId = 0;
    elFormTitle.textContent = 'Add New Domain';
    document.getElementById('submit').textContent = 'Add Domain';
    if (frm && typeof frm.reset === 'function') {
      frm.reset();
    }

    // Reset field visibility to default state (admin form only)
    if (document.getElementById('cloudflareFields')) {
      document.getElementById('cloudflareFields').style.display = 'block';
    }
    if (document.getElementById('cpanelFields')) {
      document.getElementById('cpanelFields').style.display = 'none';
    }
    if (document.getElementById('targetIpField')) {
      document.getElementById('targetIpField').style.display = 'block';
    }
    if (document.getElementById('targetCnameField')) {
      document.getElementById('targetCnameField').style.display = 'none';
    }
  }

  function render(domains){
    elTblBody.innerHTML = '';

    if (domains.length === 0) {
      const tr = document.createElement('tr');
      tr.innerHTML = '<td colspan="7" class="empty-state">No domains added yet</td>';
      elTblBody.appendChild(tr);
      return;
    }

    for (const d of domains){
      const statusBadge = d.is_active === 1
        ? '<span class=\"badge on\">Active</span>'
        : '<span class=\"badge off\">Inactive</span>';

      const wildcardBadge = d.is_wildcard === 1
        ? '<span class=\"badge admin\">Yes</span>'
        : '<span class=\"badge user\">No</span>';

      // Action buttons (admin only)
      let actionButtons = '';
      if (isAdmin) {
        actionButtons = '<div class=\"row\">'
          + '  <button class=\"btn secondary small\" data-act=\"test\" data-id=\"' + esc(d.id) + '\">Test</button>'
          + '  <button class=\"btn secondary small\" data-act=\"sync\" data-id=\"' + esc(d.id) + '\">Sync DNS</button>'
          + '  <button class=\"btn danger small\" data-act=\"delete\" data-id=\"' + esc(d.id) + '\">Delete</button>'
          + '</div>';
      } else {
        actionButtons = '<span class=\"muted\">View Only</span>';
      }

      const tr = document.createElement('tr');
      // DNS Record Type badge
      const recordType = d.target_type || 'A';
      const recordBadge = recordType === 'A'
        ? '<span class=\"badge\" style=\"background:#10b981;color:#fff;\">A</span>'
        : '<span class=\"badge\" style=\"background:#f59e0b;color:#fff;\">CNAME</span>';

      // Target (IP or CNAME)
      const target = recordType === 'A'
        ? (d.target_ip || '<span class=\"muted\">N/A</span>')
        : (d.target_cname || '<span class=\"muted\">N/A</span>');

      tr.innerHTML = ''
        + '<td><strong>' + esc(d.domain) + '</strong></td>'
        + '<td>' + recordBadge + '</td>'
        + '<td><code class=\"monospace\" style=\"font-size:12px;\">' + target + '</code></td>'
        + '<td>' + esc(d.provider.toUpperCase()) + '</td>'
        + '<td>' + wildcardBadge + '</td>'
        + '<td>' + statusBadge + '</td>'
        + '<td>' + actionButtons + '</td>';

      elTblBody.appendChild(tr);
    }
  }

  async function load(){
    try {
      const data = await apiGet('/admin/api.php?action=list_user_domains');
      if (!data || data.ok !== true){
        elMeta.textContent = '(failed to load)';
        return;
      }

      const domains = data.domains || [];
      render(domains);
      elMeta.textContent = '(' + domains.length + ')';
    } catch (e) {
      console.error('Load domains error:', e);
      elMeta.textContent = '(error)';
    }
  }

  async function onTableClick(ev){
    const t = ev.target;
    if (!(t instanceof HTMLElement)) return;

    const act = t.getAttribute('data-act') || '';
    if (!act) return;

    const id = Number(t.getAttribute('data-id') || '0');
    if (id <= 0) return;

    // Admin-only actions
    if ((act === 'delete' || act === 'sync') && !isAdmin) {
      toast.error('Permission denied: Admin only');
      return;
    }

    if (act === 'delete'){
      const confirmed = await customConfirm('Delete this domain?\\n\\nShortlinks using this domain will be affected.');
      if (!confirmed) return;

      try {
        t.disabled = true;
        t.textContent = 'Deleting...';

        const r = await apiPost('delete_user_domain', { id: id });
        if (r && r.ok === true){
          toast.success('Domain deleted');
          await load();
        } else {
          toast.error(r.error || 'Failed to delete domain');
          t.disabled = false;
          t.textContent = 'Delete';
        }
      } catch (e) {
        toast.error('Error deleting domain');
        t.disabled = false;
        t.textContent = 'Delete';
      }
      return;
    }

    if (act === 'sync'){
      try {
        t.disabled = true;
        const originalText = t.textContent;
        t.textContent = 'Syncing...';

        const r = await apiPost('sync_user_domain_dns', { id: id });
        if (r && r.ok === true){
          toast.success('DNS synced successfully');
          await load();
        } else {
          toast.error(r.error || 'Failed to sync DNS');
        }

        t.disabled = false;
        t.textContent = originalText;
      } catch (e) {
        toast.error('Error syncing DNS');
        t.disabled = false;
      }
      return;
    }

    if (act === 'test'){
      toast.info('Testing domain: Opening in new tab...');
      const testUrl = 'https://' + document.querySelectorAll('#tbl tbody tr')[Array.from(elTblBody.children).indexOf(t.closest('tr'))].querySelector('strong').textContent;
      window.open(testUrl, '_blank');
      return;
    }
  }

  // Form submission (admin only)
  if (isAdmin && frm) {
    frm.addEventListener('submit', async function(ev){
      ev.preventDefault();

      const submitBtn = document.getElementById('submit');
      const provider = document.getElementById('provider').value;
      const target_type = document.getElementById('target_type').value;

      const payload = {
        domain: (document.getElementById('domain').value || '').trim(),
        provider: provider,
        is_wildcard: document.getElementById('is_wildcard').checked ? 1 : 0,
        target_ip: target_type === 'A' ? (document.getElementById('target_ip').value || '').trim() : null,
        target_cname: target_type === 'CNAME' ? (document.getElementById('target_cname').value || '').trim() : null,
      };

      // Per-domain settings (credentials akan diambil dari Security Configuration)
      if (provider === 'cloudflare') {
        payload.cf_zone_id = (document.getElementById('cf_zone_id').value || '').trim();
        payload.cf_proxied = document.getElementById('cf_proxied').checked ? 1 : 0;
      }

      // Client validation
      if (payload.domain === '') {
        toast.error('Domain is required');
        return;
      }

      submitBtn.disabled = true;
      const originalText = submitBtn.textContent;
      submitBtn.textContent = 'Adding & Syncing DNS...';

      try {
        const r = await apiPost('create_user_domain', payload);
        if (r && r.ok === true){
          toast.success('Domain added successfully!');
          setFormModeCreate();
          await load();
          return;
        }

        toast.error(r.error || 'Failed to add domain');
      } catch (e) {
        toast.error('Network error: ' + e.message);
      } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    });

    // Reset button (admin only)
    const resetBtn = document.getElementById('reset');
    if (resetBtn) {
      resetBtn.addEventListener('click', function(){
        setFormModeCreate();
      });
    }
  }

  document.getElementById('tbl').addEventListener('click', onTableClick);

  function customConfirm(message){
    return new Promise(function(resolve){
      var overlay = document.createElement('div');
      overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.5);z-index:99998;display:flex;align-items:center;justify-content:center;';

      var dialog = document.createElement('div');
      dialog.style.cssText = 'background:#fff;padding:24px;border-radius:8px;box-shadow:0 4px 20px rgba(0,0,0,0.3);max-width:400px;width:90%;';

      var msg = document.createElement('div');
      msg.style.cssText = 'margin-bottom:20px;font-size:14px;line-height:1.5;white-space:pre-wrap;';
      msg.textContent = message;

      var btnContainer = document.createElement('div');
      btnContainer.style.cssText = 'display:flex;gap:8px;justify-content:flex-end;';

      var btnCancel = document.createElement('button');
      btnCancel.textContent = 'Cancel';
      btnCancel.className = 'btn secondary';
      btnCancel.style.cssText = 'padding:8px 16px;';

      var btnConfirm = document.createElement('button');
      btnConfirm.textContent = 'Delete';
      btnConfirm.className = 'btn';
      btnConfirm.style.cssText = 'padding:8px 16px;background:#ef4444;border-color:#ef4444;';

      btnCancel.onclick = function(){ overlay.remove(); resolve(false); };
      btnConfirm.onclick = function(){ overlay.remove(); resolve(true); };
      overlay.onclick = function(e){ if(e.target === overlay){ overlay.remove(); resolve(false); } };

      btnContainer.appendChild(btnCancel);
      btnContainer.appendChild(btnConfirm);
      dialog.appendChild(msg);
      dialog.appendChild(btnContainer);
      overlay.appendChild(dialog);
      document.body.appendChild(overlay);

      btnConfirm.focus();
    });
  }

  // DNS Guide toggle
  const toggleDnsGuideBtn = document.getElementById('toggleDnsGuide');
  const dnsGuide = document.getElementById('dnsGuide');
  if (toggleDnsGuideBtn && dnsGuide) {
    toggleDnsGuideBtn.addEventListener('click', function() {
      if (dnsGuide.style.display === 'none') {
        dnsGuide.style.display = 'block';
      } else {
        dnsGuide.style.display = 'none';
      }
    });
  }

  // Fetch server IP address
  async function fetchServerIp() {
    try {
      const serverIpEl = document.getElementById('serverIp');
      const exampleIpEl = document.getElementById('exampleIp');
      const exampleIp2El = document.getElementById('exampleIp2');

      // Fetch IP from ipify API
      const response = await fetch('https://api.ipify.org?format=json');
      const data = await response.json();

      if (data && data.ip) {
        const serverIp = data.ip;
        if (serverIpEl) serverIpEl.textContent = serverIp;
        if (exampleIpEl) exampleIpEl.textContent = serverIp;
        if (exampleIp2El) exampleIp2El.textContent = serverIp;
      } else {
        if (serverIpEl) serverIpEl.textContent = 'Unable to detect';
      }
    } catch (e) {
      const serverIpEl = document.getElementById('serverIp');
      if (serverIpEl) serverIpEl.textContent = 'Unable to detect';
      console.error('Failed to fetch server IP:', e);
    }
  }

  setFormModeCreate();
  load();
  fetchServerIp();
})();
</script>
</body>
</html>
