<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';
\App\Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();
Auth::requireAdmin(); // Admin only!
$csrf = Auth::ensureCsrfToken();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <title>Global Domains | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Global Domains</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <a href="/admin/users.php">Users</a>
        <a href="/admin/domains.php">Domains</a>
        <a href="/admin/global-domains.php" aria-current="page">Global Domains</a>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">Global Domains</h1>
            <div class="page-sub">Manage domains accessible to ALL users</div>
          </div>
        </div>

        <!-- DNS Setup Guide -->
        <div class="grid">
          <div class="card full">
            <div class="card-head">
              <div>
                <h3 class="card-title">DNS Setup Guide (Global Domains)</h3>
                <div class="card-sub">Configure DNS records for global domains</div>
              </div>
              <div class="right">
                <button class="btn btn-sm" id="toggleDnsGuide" type="button">Show / Hide</button>
              </div>
            </div>

            <div id="dnsGuide" style="display: none; margin-top: 10px;">
              <div class="warning-box">
                <strong>Global domains are available to ALL users.</strong> Configure DNS to point to this server.
              </div>

              <h4 style="font-size: 0.9rem; font-weight: 600; margin: 12px 0 8px;">1. DNS Record Types</h4>

              <div class="table-wrap">
                <table>
                  <thead>
                    <tr>
                      <th style="width:120px;">Record Type</th>
                      <th>Description</th>
                      <th style="width:200px;">Example</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr>
                      <td><span class="tag">A Record</span></td>
                      <td>Points domain directly to an IP address. Use this if you know the server IP.</td>
                      <td><code style="font-size:0.78rem;">192.0.2.1</code></td>
                    </tr>
                    <tr>
                      <td><span class="tag">CNAME</span></td>
                      <td>Points domain to another domain name (alias). Use this for CDN or load balancers.</td>
                      <td><code style="font-size:0.78rem;">server.example.com</code></td>
                    </tr>
                  </tbody>
                </table>
              </div>

              <h4 style="font-size: 0.9rem; font-weight: 600; margin: 12px 0 8px;">2. Get Server Information</h4>

              <div style="display: flex; gap: 12px; margin-bottom: 12px;">
                <div style="flex: 1;">
                  <div class="muted" style="margin-bottom: 4px; font-size: 0.78rem;">Server IP Address:</div>
                  <code id="serverIp" style="font-size: 0.82rem; background: #f9fafb; padding: 8px 12px; border-radius: var(--radius); display: block; border: 1px solid var(--border-color);">Loading...</code>
                </div>
                <div style="flex: 1;">
                  <div class="muted" style="margin-bottom: 4px; font-size: 0.78rem;">Server Hostname:</div>
                  <code style="font-size: 0.82rem; background: #f9fafb; padding: 8px 12px; border-radius: var(--radius); display: block; border: 1px solid var(--border-color);"><?= e($_SERVER['HTTP_HOST'] ?? 'N/A') ?></code>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="grid">
          <!-- Form Card -->
          <div class="card">
            <div class="card-head">
              <h3 class="card-title" id="formTitle">Add Global Domain</h3>
            </div>

            <form id="frm" autocomplete="off">
              <div>
                <label for="domain">Domain / Subdomain</label>
                <input type="text" id="domain" name="domain" placeholder="go.link or short.example.com" required>
                <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">This domain will be available to ALL users</div>
              </div>

              <div style="margin-top: 12px;">
                <label for="provider">DNS Provider</label>
                <select id="provider" name="provider" required>
                  <option value="cloudflare" selected>Cloudflare DNS</option>
                  <option value="cpanel">cPanel</option>
                </select>

                <div class="warning-box" style="margin-top: 8px;">
                  <strong>ℹ️ Note:</strong> API credentials akan diambil dari <a href="/admin/settings.php" style="color: #0066cc; text-decoration: underline;">Security Configuration</a>. Pastikan sudah mengkonfigurasi Cloudflare atau cPanel API credentials di Settings.
                </div>

                <div id="cloudflareFields" style="display: block; margin-top: 12px;">
                  <label for="cf_zone_id">Zone ID (Optional)</label>
                  <input id="cf_zone_id" name="cf_zone_id" placeholder="Auto-detect if empty">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Leave empty to auto-detect from Cloudflare API</div>

                  <div style="margin-top: 12px;">
                    <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                      <input type="checkbox" id="cf_proxied" name="cf_proxied" value="1" style="width: 16px; height: 16px;">
                      <span style="font-size: 0.82rem;">Enable Cloudflare Proxy (Orange Cloud)</span>
                    </label>
                  </div>
                </div>

                <div id="cpanelFields" style="display: none; margin-top: 12px;">
                  <div class="muted" style="font-size: 0.82rem;">
                    cPanel credentials akan diambil dari Security Configuration.
                  </div>
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label for="target_type">DNS Record Type</label>
                <select id="target_type" name="target_type" required>
                  <option value="A" selected>A Record (IP Address)</option>
                  <option value="CNAME">CNAME Record (Domain)</option>
                </select>

                <div id="targetIpField" style="display: block; margin-top: 12px;">
                  <label for="target_ip">Target IP Address</label>
                  <input id="target_ip" name="target_ip" placeholder="192.0.2.1">
                </div>

                <div id="targetCnameField" style="display: none; margin-top: 12px;">
                  <label for="target_cname">Target CNAME</label>
                  <input id="target_cname" name="target_cname" placeholder="example.com">
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="is_wildcard" name="is_wildcard" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem; font-weight: 500;">Enable Wildcard DNS (*)</span>
                </label>
                <div class="muted" style="margin-top: 6px; font-size: 0.78rem;">
                  Allows subdomains like: code1.yourdomain.com, code2.yourdomain.com
                </div>
              </div>

              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="is_default" name="is_default" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem; font-weight: 500;">Set as Default Domain</span>
                </label>
                <div class="muted" style="margin-top: 6px; font-size: 0.78rem;">
                  Default domain akan auto-selected saat user create shortlink baru
                </div>
              </div>

              <div style="margin-top: 16px; display: flex; gap: 8px;">
                <button class="btn" id="submit" type="submit">Add Domain</button>
                <button class="btn btn-secondary" id="reset" type="button">Reset</button>
              </div>
            </form>
          </div>

          <!-- Table Card -->
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Global Domains</h3>
                <div class="card-sub" id="meta"></div>
              </div>
            </div>

            <div class="table-wrap">
              <table id="tbl">
                <thead>
                  <tr>
                    <th>Domain</th>
                    <th style="width:100px;">DNS Type</th>
                    <th style="width:150px;">Target</th>
                    <th style="width:100px;">Provider</th>
                    <th style="width:80px;">Default</th>
                    <th style="width:100px;">Status</th>
                    <th style="width:180px;">Actions</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

<script nonce="<?= e($nonce) ?>">
(function(){
  'use strict';

  const csrf = document.querySelector('meta[name=\"csrf-token\"]').getAttribute('content') || '';
  let editingId = 0;

  const elTblBody = document.querySelector('#tbl tbody');
  const elMeta = document.getElementById('meta');
  const elFormTitle = document.getElementById('formTitle');
  const frm = document.getElementById('frm');

  document.getElementById('provider').addEventListener('change', function(){
    const provider = this.value;
    document.getElementById('cloudflareFields').style.display = provider === 'cloudflare' ? 'block' : 'none';
    document.getElementById('cpanelFields').style.display = provider === 'cpanel' ? 'block' : 'none';
  });

  document.getElementById('target_type').addEventListener('change', function(){
    const type = this.value;
    document.getElementById('targetIpField').style.display = type === 'A' ? 'block' : 'none';
    document.getElementById('targetCnameField').style.display = type === 'CNAME' ? 'block' : 'none';
  });

  if (typeof window.toast === 'undefined') {
    window.toast = {
      _show: function(msg, type){
        var toast = document.createElement('div');
        toast.style.cssText = 'position:fixed;top:20px;right:20px;background:#fff;border-left:4px solid ' +
          (type === 'error' ? '#ef4444' : type === 'success' ? '#10b981' : '#3b82f6') +
          ';box-shadow:0 4px 12px rgba(0,0,0,0.15);padding:16px 20px;border-radius:4px;z-index:99999;max-width:400px;';
        toast.textContent = (type === 'error' ? '✗ ' : type === 'success' ? '✓ ' : 'ℹ ') + msg;
        document.body.appendChild(toast);
        setTimeout(function(){ toast.style.transition = 'opacity 0.3s'; toast.style.opacity = '0'; setTimeout(function(){ toast.remove(); }, 300); }, 4000);
      },
      success: function(msg){ this._show(msg, 'success'); },
      error: function(msg){ this._show(msg, 'error'); },
      info: function(msg){ this._show(msg, 'info'); }
    };
  }

  function esc(s){
    return String(s).replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;').replaceAll('"','&quot;').replaceAll("'",'&#039;');
  }

  async function apiPost(action, payload){
    const res = await fetch('/admin/api.php?action=' + encodeURIComponent(action), {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/json', 'X-CSRF-Token': csrf },
      body: JSON.stringify(payload || {})
    });
    return await res.json();
  }

  async function apiGet(url){
    const res = await fetch(url, { credentials: 'same-origin' });
    return await res.json();
  }

  function setFormModeCreate(){
    editingId = 0;
    elFormTitle.textContent = 'Add Global Domain';
    document.getElementById('submit').textContent = 'Add Domain';
    if (frm && typeof frm.reset === 'function') frm.reset();

    // Reset field visibility to default state
    document.getElementById('cloudflareFields').style.display = 'block';
    document.getElementById('cpanelFields').style.display = 'none';
    document.getElementById('targetIpField').style.display = 'block';
    document.getElementById('targetCnameField').style.display = 'none';
  }

  function render(domains){
    elTblBody.innerHTML = '';

    if (domains.length === 0) {
      const tr = document.createElement('tr');
      tr.innerHTML = '<td colspan="7" class="empty-state">No global domains configured</td>';
      elTblBody.appendChild(tr);
      return;
    }

    for (const d of domains){
      const statusBadge = d.is_active === 1
        ? '<span class=\"badge on\">Active</span>'
        : '<span class=\"badge off\">Inactive</span>';

      const defaultBadge = d.is_default === 1
        ? '<span class=\"badge\" style=\"background:#10b981;color:#fff;\">✓ Default</span>'
        : '';

      const tr = document.createElement('tr');
      // DNS Record Type badge
      const recordType = d.target_type || 'A';
      const recordBadge = recordType === 'A'
        ? '<span class=\"badge\" style=\"background:#10b981;color:#fff;\">A</span>'
        : '<span class=\"badge\" style=\"background:#f59e0b;color:#fff;\">CNAME</span>';

      // Target (IP or CNAME)
      const target = recordType === 'A'
        ? (d.target_ip || '<span class=\"muted\">N/A</span>')
        : (d.target_cname || '<span class=\"muted\">N/A</span>');

      tr.innerHTML = ''
        + '<td><strong>' + esc(d.domain) + '</strong><br><span class=\"muted\" style=\"font-size:11px;\">' + (d.is_wildcard === 1 ? 'Wildcard: *.'+esc(d.domain) : 'Single domain') + '</span></td>'
        + '<td>' + recordBadge + '</td>'
        + '<td><code class=\"monospace\" style=\"font-size:12px;\">' + target + '</code></td>'
        + '<td>' + esc(d.provider.toUpperCase()) + '</td>'
        + '<td>' + defaultBadge + '</td>'
        + '<td>' + statusBadge + '</td>'
        + '<td>'
        + '  <div class=\"row\">'
        + '    <button class=\"btn secondary small\" data-act=\"sync\" data-id=\"' + esc(d.id) + '\">Sync DNS</button>'
        + '    <button class=\"btn danger small\" data-act=\"delete\" data-id=\"' + esc(d.id) + '\">Delete</button>'
        + '  </div>'
        + '</td>';

      elTblBody.appendChild(tr);
    }
  }

  async function load(){
    try {
      const data = await apiGet('/admin/api.php?action=list_global_domains');
      if (!data || data.ok !== true){
        elMeta.textContent = '(failed to load)';
        return;
      }

      const domains = data.domains || [];
      render(domains);
      elMeta.textContent = '(' + domains.length + ')';
    } catch (e) {
      console.error('Load error:', e);
      elMeta.textContent = '(error)';
    }
  }

  async function onTableClick(ev){
    const t = ev.target;
    if (!(t instanceof HTMLElement)) return;

    const act = t.getAttribute('data-act') || '';
    if (!act) return;

    const id = Number(t.getAttribute('data-id') || '0');
    if (id <= 0) return;

    if (act === 'delete'){
      const confirmed = await customConfirm('Delete this global domain?\\n\\nAll users will lose access to this domain.');
      if (!confirmed) return;

      try {
        t.disabled = true;
        t.textContent = 'Deleting...';

        const r = await apiPost('delete_global_domain', { id: id });
        if (r && r.ok === true){
          toast.success('Global domain deleted');
          await load();
        } else {
          toast.error(r.error || 'Failed to delete');
          t.disabled = false;
          t.textContent = 'Delete';
        }
      } catch (e) {
        toast.error('Error deleting domain');
        t.disabled = false;
        t.textContent = 'Delete';
      }
      return;
    }

    if (act === 'sync'){
      try {
        t.disabled = true;
        const originalText = t.textContent;
        t.textContent = 'Syncing...';

        const r = await apiPost('sync_global_domain_dns', { id: id });
        if (r && r.ok === true){
          toast.success('DNS synced successfully');
          await load();
        } else {
          toast.error(r.error || 'Failed to sync DNS');
        }

        t.disabled = false;
        t.textContent = originalText;
      } catch (e) {
        toast.error('Error syncing DNS');
        t.disabled = false;
      }
      return;
    }
  }

  frm.addEventListener('submit', async function(ev){
    ev.preventDefault();

    const submitBtn = document.getElementById('submit');
    const provider = document.getElementById('provider').value;
    const target_type = document.getElementById('target_type').value;

    const payload = {
      domain: (document.getElementById('domain').value || '').trim(),
      provider: provider,
      is_wildcard: document.getElementById('is_wildcard').checked ? 1 : 0,
      is_default: document.getElementById('is_default').checked ? 1 : 0,
      target_ip: target_type === 'A' ? (document.getElementById('target_ip').value || '').trim() : null,
      target_cname: target_type === 'CNAME' ? (document.getElementById('target_cname').value || '').trim() : null,
    };

    // Per-domain settings (credentials akan diambil dari Security Configuration)
    if (provider === 'cloudflare') {
      payload.cf_zone_id = (document.getElementById('cf_zone_id').value || '').trim();
      payload.cf_proxied = document.getElementById('cf_proxied').checked ? 1 : 0;
    }

    if (payload.domain === '') {
      toast.error('Domain is required');
      return;
    }

    submitBtn.disabled = true;
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Adding & Syncing DNS...';

    try {
      const r = await apiPost('create_global_domain', payload);
      if (r && r.ok === true){
        toast.success('Global domain added!');
        setFormModeCreate();
        await load();
        return;
      }

      toast.error(r.error || 'Failed to add domain');
    } catch (e) {
      toast.error('Network error: ' + e.message);
    } finally {
      submitBtn.disabled = false;
      submitBtn.textContent = originalText;
    }
  });

  document.getElementById('reset').addEventListener('click', function(){
    setFormModeCreate();
  });

  document.getElementById('tbl').addEventListener('click', onTableClick);

  function customConfirm(message){
    return new Promise(function(resolve){
      var overlay = document.createElement('div');
      overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.5);z-index:99998;display:flex;align-items:center;justify-content:center;';
      var dialog = document.createElement('div');
      dialog.style.cssText = 'background:#fff;padding:24px;border-radius:8px;box-shadow:0 4px 20px rgba(0,0,0,0.3);max-width:400px;width:90%;';
      var msg = document.createElement('div');
      msg.style.cssText = 'margin-bottom:20px;font-size:14px;line-height:1.5;white-space:pre-wrap;';
      msg.textContent = message;
      var btnContainer = document.createElement('div');
      btnContainer.style.cssText = 'display:flex;gap:8px;justify-content:flex-end;';
      var btnCancel = document.createElement('button');
      btnCancel.textContent = 'Cancel';
      btnCancel.className = 'btn secondary';
      btnCancel.style.cssText = 'padding:8px 16px;';
      var btnConfirm = document.createElement('button');
      btnConfirm.textContent = 'Delete';
      btnConfirm.className = 'btn';
      btnConfirm.style.cssText = 'padding:8px 16px;background:#ef4444;border-color:#ef4444;';
      btnCancel.onclick = function(){ overlay.remove(); resolve(false); };
      btnConfirm.onclick = function(){ overlay.remove(); resolve(true); };
      overlay.onclick = function(e){ if(e.target === overlay){ overlay.remove(); resolve(false); } };
      btnContainer.appendChild(btnCancel);
      btnContainer.appendChild(btnConfirm);
      dialog.appendChild(msg);
      dialog.appendChild(btnContainer);
      overlay.appendChild(dialog);
      document.body.appendChild(overlay);
      btnConfirm.focus();
    });
  }

  // DNS Guide toggle
  const toggleDnsGuideBtn = document.getElementById('toggleDnsGuide');
  const dnsGuide = document.getElementById('dnsGuide');
  if (toggleDnsGuideBtn && dnsGuide) {
    toggleDnsGuideBtn.addEventListener('click', function() {
      if (dnsGuide.style.display === 'none') {
        dnsGuide.style.display = 'block';
      } else {
        dnsGuide.style.display = 'none';
      }
    });
  }

  // Fetch server IP address
  async function fetchServerIp() {
    try {
      const serverIpEl = document.getElementById('serverIp');
      const response = await fetch('https://api.ipify.org?format=json');
      const data = await response.json();

      if (data && data.ip) {
        if (serverIpEl) serverIpEl.textContent = data.ip;
      } else {
        if (serverIpEl) serverIpEl.textContent = 'Unable to detect';
      }
    } catch (e) {
      const serverIpEl = document.getElementById('serverIp');
      if (serverIpEl) serverIpEl.textContent = 'Unable to detect';
      console.error('Failed to fetch server IP:', e);
    }
  }

  setFormModeCreate();
  load();
  fetchServerIp();
})();
</script>
</body>
</html>
