<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';

Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();
$csrf = Auth::ensureCsrfToken();
$isAdmin = Auth::isAdmin();

$publicBase = trim(Env::getString('PUBLIC_BASE_URL', ''));
$publicBase = rtrim($publicBase, '/');

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <meta name="public-base" content="<?= e($publicBase) ?>">
  <title>Dashboard | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Dashboard</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php" aria-current="page">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/users.php">Users</a>
        <?php endif; ?>
        <a href="/admin/domains.php">Domains</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/global-domains.php">Global Domains</a>
        <?php endif; ?>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">Dashboard</h1>
            <div class="page-sub">Sinkron ke table <span style="font-family: monospace; background: var(--bg-secondary); padding: 2px 6px; border-radius: 3px; font-size: 0.85em;">shortlinks</span>. Tidak ada sihir, cuma SQL.</div>
          </div>
        </div>

        <div class="grid">
          <!-- Form Card -->
          <div class="card">
            <div class="card-head">
              <h3 class="card-title" id="formTitle">Create Shortlink</h3>
            </div>

            <form id="frm" autocomplete="off">
              <!-- 1. 3rd Party & Domain - Side by side -->
              <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 12px;">
                <div>
                  <label for="third_party_service">3rd Party Shortener</label>
                  <select id="third_party_service" name="third_party_service">
                    <option value="">None (Use Internal)</option>
                    <option value="isgd">is.gd</option>
                    <option value="vgd">v.gd</option>
                    <option value="tinyurl">TinyURL</option>
                    <option value="dagd">da.gd</option>
                    <option value="clckru">clck.ru</option>
                  </select>
                </div>

                <div>
                  <label for="domain_selector">Domain</label>
                  <select id="domain_selector" name="domain_selector">
                    <option value="">Loading domains...</option>
                  </select>
                </div>
              </div>

              <!-- 2. Custom Code -->
              <input type="hidden" id="code" name="code" value="">

              <!-- 3. OG Title -->
              <div style="margin-top: 12px;">
                <label for="og_title"></label>
                <input id="og_title" name="og_title" placeholder="Page Title">
              </div>

              <!-- 4. OG Description -->
              <div style="margin-top: 12px;">
                <label for="og_description"></label>
                <textarea id="og_description" name="og_description" placeholder="Page description..." rows="3"></textarea>
              </div>

              <!-- 5. OG Image -->
              <div style="margin-top: 12px;">
                <label for="og_image_url"></label>
                <input id="og_image_url" name="og_image_url" placeholder="https://example.com/image.jpg">
              </div>

              <!-- 6. Enable Shim Link -->
              <div style="margin-top: 12px;">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                  <input type="checkbox" id="use_shim" name="use_shim" value="1" style="width: 16px; height: 16px;">
                  <span style="font-size: 0.82rem;">Enable Shim Link</span>
                </label>
              </div>

              <!-- 7. Jumlah Shortlink | Simpan | Reset - Sejajar -->
              <div style="margin-top: 12px; display: grid; grid-template-columns: 100px 1fr 1fr; gap: 8px; align-items: end;">
                <div>
                  <label for="bulk_count">Limit</label>
                  <input id="bulk_count" name="bulk_count" type="number" min="1" max="25" value="1">
                </div>

                <button class="btn" id="submit" type="submit">Simpan</button>
                <button class="btn btn-secondary" id="reset" type="button">Reset</button>
              </div>

              <!-- Hidden fields - Auto-filled from settings -->
              <input type="hidden" id="og_site_name" name="og_site_name" value="">
              <input type="hidden" id="og_locale" name="og_locale" value="id_ID">
              <input type="hidden" id="og_type" name="og_type" value="website">

              <!-- 8. Output Textarea (Copy di dalam textarea) -->
              <div style="margin-top: 12px;">
                <label for="bulk_output">Output (Generated Links)</label>

                <div style="position: relative; margin-top: 6px;">
                  <textarea
                    id="bulk_output"
                    name="bulk_output"
                    readonly
                    placeholder="Generated shortlinks will appear here..."
                    style="font-family: monospace; padding-right: 80px; min-height: 100px;"
                  ></textarea>

                  <button
                    type="button"
                    id="copyOutputBtn"
                    class="btn btn-sm btn-secondary"
                    aria-label="Copy generated links"
                    style="position: absolute; top: 8px; right: 8px;"
                  >
                    Copy
                  </button>
                </div>
              </div>
            </form>
          </div>

          <!-- Table Card -->
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Shortlink List</h3>
                <div class="card-sub" id="meta"></div>
              </div>
            </div>

            <!-- Bulk Actions Bar -->
            <div style="display: flex; gap: 8px; margin-bottom: 12px; display: none;" id="bulkActionsBar">
              <button class="btn btn-danger" id="bulkDeleteBtn" type="button">
                Delete Selected <span id="selectedCount">0</span>
              </button>
              <button class="btn btn-secondary" id="clearSelectionBtn" type="button">Clear Selection</button>
            </div>

            <div class="table-wrap">
              <table id="tbl">
                <thead>
                  <tr>
                    <th style="width: 40px;">
                      <input type="checkbox" id="selectAll" style="cursor: pointer;">
                    </th>
                    <th>URL</th>
                    <th>Title</th>
                    <th style="width: 100px;">Status</th>
                    <th style="width: 140px;">Updated</th>
                    <th style="width: 180px;">Actions</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>

            <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 12px;">
              <button class="btn btn-secondary" id="prev" type="button">Prev</button>
              <div class="muted" id="pageInfo"></div>
              <button class="btn btn-secondary" id="next" type="button">Next</button>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

<script nonce="<?= e($nonce) ?>">
(function () {
  'use strict';

  function byId(id) {
    return document.getElementById(id);
  }

  function getMeta(name) {
    var el = document.querySelector('meta[name="' + name + '"]');
    return el ? (el.getAttribute('content') || '') : '';
  }

  var csrf = getMeta('csrf-token');
  var base = getMeta('public-base').replace(/\/+$/g, '');

  var page = 1;
  var pageSize = 20;
  var editingCode = '';

  // Store available domains for random selection
  var availableDomains = [];

  var elTblBody = document.querySelector('#tbl tbody');
  var elMeta = byId('meta');
  var elPageInfo = byId('pageInfo');
  var elFormTitle = byId('formTitle');
  var frm = byId('frm');

  // Ensure toast is available
  if (typeof window.toast === 'undefined') {
    window.toast = {
      _show: function (msg, type) {
        var toastEl = document.createElement('div');
        toastEl.className = 'toast toast-' + type;
        toastEl.textContent = (type === 'error' ? '✗ ' : (type === 'success' ? '✓ ' : 'ℹ ')) + msg;
        document.body.appendChild(toastEl);
        setTimeout(function () {
          toastEl.classList.add('toast-fadeout');
          setTimeout(function () { toastEl.remove(); }, 300);
        }, 4000);
      },
      success: function (msg) { this._show(msg, 'success'); },
      error: function (msg) { this._show(msg, 'error'); },
      info: function (msg) { this._show(msg, 'info'); }
    };
  }

  function esc(s) {
    return String(s)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function shortUrl(code) {
    if (!base) return '/' + code;
    return base.replace(/\/+$/g, '') + '/' + code;
  }

  async function safeJsonResponse(res) {
    var ct = (res.headers && res.headers.get && res.headers.get('content-type')) ? res.headers.get('content-type') : '';
    var text = '';
    try {
      text = await res.text();
    } catch (e) {
      return { ok: false, error: 'Failed to read response' };
    }

    if (!ct || ct.indexOf('application/json') === -1) {
      return { ok: false, error: 'Non-JSON response', raw: text };
    }

    try {
      return JSON.parse(text);
    } catch (e) {
      return { ok: false, error: 'Invalid JSON response' };
    }
  }

  async function apiGet(url) {
    var res;
    try {
      res = await fetch(url, { credentials: 'same-origin' });
    } catch (e) {
      return { ok: false, error: 'Network error' };
    }

    var data = await safeJsonResponse(res);

    if (!res.ok && data && typeof data === 'object' && !data.error) {
      data.error = 'HTTP ' + String(res.status);
    }

    return data;
  }

  async function apiPost(action, payload) {
    var res;
    try {
      res = await fetch('/admin/api.php?action=' + encodeURIComponent(action), {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'X-CSRF-Token': csrf
        },
        body: JSON.stringify(payload || {})
      });
    } catch (e) {
      return { ok: false, error: 'Network error' };
    }

    var data = await safeJsonResponse(res);

    if (!res.ok && data && typeof data === 'object' && !data.error) {
      data.error = 'HTTP ' + String(res.status);
    }

    return data;
  }

  async function createThirdPartyShortlink(service, longUrl) {
    var payload = { service: service, long_url: longUrl };
    var data = await apiPost('create_third_party_shortlink', payload);

    if (data && data.ok === true && data.url) {
      return { ok: true, url: data.url };
    }

    return { ok: false, error: (data && data.error) ? data.error : '3rd party API error' };
  }

  function validateCode(code, allowEmpty) {
    var v = String(code || '').trim();
    if (v === '') return allowEmpty === true;
    // 3-64 chars, starts with alnum, allows _ and -
    return /^[A-Za-z0-9][A-Za-z0-9_-]{2,63}$/.test(v);
  }

  function validateOg(title, desc, imgUrl) {
    var t = String(title || '').trim();
    var d = String(desc || '').trim();
    var u = String(imgUrl || '').trim();

    if (t.length > 120) return { ok: false, error: 'OG Title terlalu panjang (max 120)' };
    if (d.length > 400) return { ok: false, error: 'OG Description terlalu panjang (max 400)' };

    if (u !== '') {
      if (u.length > 2048) return { ok: false, error: 'OG Image URL terlalu panjang (max 2048)' };
      try {
        var parsed = new URL(u);
        if (parsed.protocol !== 'https:' && parsed.protocol !== 'http:') {
          return { ok: false, error: 'OG Image URL harus http/https' };
        }
      } catch (e) {
        return { ok: false, error: 'OG Image URL tidak valid' };
      }
    }

    return { ok: true };
  }

  // Custom confirm dialog
  function customConfirm(message) {
    return new Promise(function (resolve) {
      var overlay = document.createElement('div');
      overlay.className = 'dialog-overlay';

      var dialog = document.createElement('div');
      dialog.className = 'dialog';

      var msg = document.createElement('div');
      msg.className = 'dialog-message';
      msg.textContent = message;

      var btnContainer = document.createElement('div');
      btnContainer.className = 'dialog-buttons';

      var btnCancel = document.createElement('button');
      btnCancel.textContent = 'Cancel';
      btnCancel.className = 'btn secondary dialog-button';
      btnCancel.type = 'button';

      var btnConfirm = document.createElement('button');
      btnConfirm.textContent = 'Delete';
      btnConfirm.className = 'btn btn-bulk-delete dialog-button';
      btnConfirm.type = 'button';

      btnCancel.onclick = function () { overlay.remove(); resolve(false); };
      btnConfirm.onclick = function () { overlay.remove(); resolve(true); };
      overlay.onclick = function (ev) { if (ev.target === overlay) { overlay.remove(); resolve(false); } };

      btnContainer.appendChild(btnCancel);
      btnContainer.appendChild(btnConfirm);
      dialog.appendChild(msg);
      dialog.appendChild(btnContainer);
      overlay.appendChild(dialog);
      document.body.appendChild(overlay);

      btnConfirm.focus();
    });
  }

  async function loadSettings() {
    var data = await apiGet('/admin/api.php?action=get_settings');
    if (data && data.ok === true && data.settings) {
      byId('og_site_name').value = data.settings.default_site_name || '';
      byId('og_locale').value = data.settings.default_locale || 'id_ID';
      byId('og_type').value = data.settings.default_og_type || 'website';
    }
  }

  async function setFormModeCreate() {
    editingCode = '';
    elFormTitle.textContent = 'Create Shortlink';

    byId('code').disabled = false;
    byId('bulk_count').disabled = false;
    byId('bulk_count').value = '1';
    byId('bulk_output').value = '';
    byId('submit').textContent = 'Simpan';

    if (frm && typeof frm.reset === 'function') {
      frm.reset();
    } else {
      byId('code').value = '';
      byId('og_title').value = '';
      byId('og_description').value = '';
      byId('og_image_url').value = '';
      byId('use_shim').checked = false;
    }

    byId('domain_selector').value = '';

    await loadSettings();
  }

  async function fillForm(row) {
    editingCode = row.code;
    elFormTitle.textContent = 'Edit: ' + row.code;

    byId('code').value = row.code;
    byId('code').disabled = true;

    byId('bulk_count').disabled = true;
    byId('bulk_count').value = '1';
    byId('bulk_output').value = '';

    byId('og_title').value = row.og_title || '';
    byId('og_description').value = row.og_description || '';
    byId('og_image_url').value = row.og_image_url || '';
    byId('og_type').value = row.og_type || 'website';
    byId('og_site_name').value = row.og_site_name || '';
    byId('og_locale').value = row.og_locale || 'id_ID';
    byId('use_shim').checked = (row.use_shim === 1);

    var domainSelector = byId('domain_selector');
    if (row.domain_type && row.domain_id) {
      domainSelector.value = row.domain_type + ':' + row.domain_id;
    } else {
      domainSelector.value = '';
    }

    byId('submit').textContent = 'Update';
  }

  function render(rows) {
    elTblBody.innerHTML = '';

    if (!rows || rows.length === 0) {
      var trEmpty = document.createElement('tr');
      trEmpty.innerHTML = '<td colspan="6" class="empty-state">No shortlinks found</td>';
      elTblBody.appendChild(trEmpty);
      updateSelectionUI();
      return;
    }

    for (var i = 0; i < rows.length; i++) {
      var r = rows[i];

      var badge = r.is_active === 1 ? '<span class="badge on">ON</span>' : '<span class="badge off">OFF</span>';

      var updatedAt = '';
      try {
        updatedAt = new Date(r.updated_at).toLocaleString('id-ID', {
          day: '2-digit',
          month: 'short',
          year: 'numeric',
          hour: '2-digit',
          minute: '2-digit'
        });
      } catch (e) {
        updatedAt = '';
      }

      var fullUrl = '';
      if (r.domain_name) {
        fullUrl = 'https://' + String(r.domain_name) + '/' + String(r.code);
      } else {
        fullUrl = shortUrl(String(r.code));
      }

      var tr = document.createElement('tr');
      tr.innerHTML = ''
        + '<td class="text-center">'
        + '  <input type="checkbox" class="row-select cursor-pointer" data-code="' + esc(r.code) + '">'
        + '</td>'
        + '<td><a href="' + esc(fullUrl) + '" target="_blank" rel="noopener" class="link-primary">' + esc(fullUrl) + '</a></td>'
        + '<td>' + esc(r.og_title || '') + '</td>'
        + '<td>' + badge + '</td>'
        + '<td class="text-muted-sm">' + esc(updatedAt) + '</td>'
        + '<td>'
        + '  <div class="row row-gap-sm">'
        + '    <a class="btn secondary small" href="/admin/analytics.php?code=' + encodeURIComponent(r.code) + '">Analytics</a>'
        + '    <button class="btn secondary small" type="button" data-act="edit" data-code="' + esc(r.code) + '">Edit</button>'
        + '    <button class="btn secondary small danger" type="button" data-act="delete" data-code="' + esc(r.code) + '">Delete</button>'
        + '  </div>'
        + '</td>';

      elTblBody.appendChild(tr);
    }

    updateSelectionUI();
  }

  async function load() {
    var url = '/admin/api.php?action=list'
      + '&q='
      + '&status=all'
      + '&page=' + encodeURIComponent(String(page))
      + '&page_size=' + encodeURIComponent(String(pageSize));

    var data = await apiGet(url);

    if (!data || data.ok !== true) {
      elMeta.textContent = 'Failed to load';
      toast.error((data && data.error) ? data.error : 'Failed to load shortlinks');
      render([]);
      return;
    }

    render(data.rows || []);
    elMeta.textContent = 'Total: ' + String(data.total || 0);
    elPageInfo.textContent = 'Page ' + String(page);
  }

  // Selection Management
  function getSelectedCodes() {
    var checkboxes = document.querySelectorAll('.row-select:checked');
    var arr = [];
    for (var i = 0; i < checkboxes.length; i++) {
      var v = checkboxes[i].getAttribute('data-code');
      if (v) arr.push(v);
    }
    return arr;
  }

  function updateSelectionUI() {
    var selectedCodes = getSelectedCodes();
    var count = selectedCodes.length;

    var selectedCountEl = byId('selectedCount');
    if (selectedCountEl) selectedCountEl.textContent = String(count);

    var bulkActionsBar = byId('bulkActionsBar');
    if (bulkActionsBar) {
      if (count > 0) {
        bulkActionsBar.classList.remove('hidden');
      } else {
        bulkActionsBar.classList.add('hidden');
      }
    }

    var allCheckboxes = document.querySelectorAll('.row-select');
    var selectAllCheckbox = byId('selectAll');
    if (selectAllCheckbox) {
      if (allCheckboxes.length > 0) {
        selectAllCheckbox.checked = (allCheckboxes.length === count);
        selectAllCheckbox.indeterminate = (count > 0 && count < allCheckboxes.length);
      } else {
        selectAllCheckbox.checked = false;
        selectAllCheckbox.indeterminate = false;
      }
    }
  }

  function clearSelection() {
    var checked = document.querySelectorAll('.row-select:checked');
    for (var i = 0; i < checked.length; i++) {
      checked[i].checked = false;
    }
    updateSelectionUI();
  }

  async function bulkDelete() {
    var selectedCodes = getSelectedCodes();
    if (selectedCodes.length === 0) {
      toast.error('No shortlinks selected');
      return;
    }

    var preview = selectedCodes.slice(0, 20).join(', ');
    var suffix = selectedCodes.length > 20 ? ', ...' : '';
    var confirmed = await customConfirm(
      'Delete ' + selectedCodes.length + ' shortlink(s)?\n\n' +
      'Codes: ' + preview + suffix + '\n\n' +
      'This action cannot be undone.'
    );

    if (!confirmed) return;

    var bulkDeleteBtn = byId('bulkDeleteBtn');
    if (!bulkDeleteBtn) return;

    bulkDeleteBtn.disabled = true;

    var result = await apiPost('bulk_delete', { codes: selectedCodes });

    if (result && result.ok === true) {
      toast.success('Deleted ' + String(result.deleted || 0) + ' shortlink(s)');
      clearSelection();
      await load();
    } else {
      toast.error((result && result.error) ? result.error : 'Bulk delete failed');
    }

    bulkDeleteBtn.disabled = false;
  }

  var selectAllEl = byId('selectAll');
  if (selectAllEl) {
    selectAllEl.addEventListener('change', function () {
      var checkboxes = document.querySelectorAll('.row-select');
      for (var i = 0; i < checkboxes.length; i++) {
        checkboxes[i].checked = this.checked;
      }
      updateSelectionUI();
    });
  }

  var tblEl = byId('tbl');
  if (tblEl) {
    tblEl.addEventListener('change', function (ev) {
      var t = ev.target;
      if (t && t.classList && t.classList.contains('row-select')) {
        updateSelectionUI();
      }
    });
  }

  var clearSelBtn = byId('clearSelectionBtn');
  if (clearSelBtn) clearSelBtn.addEventListener('click', clearSelection);

  var bulkDeleteBtn = byId('bulkDeleteBtn');
  if (bulkDeleteBtn) bulkDeleteBtn.addEventListener('click', bulkDelete);

  async function onTableClick(ev) {
    var t = ev.target;
    if (!t || !(t instanceof HTMLElement)) return;

    var act = t.getAttribute('data-act') || '';
    if (!act) return;

    var code = t.getAttribute('data-code') || '';
    if (!code) return;

    if (act === 'edit') {
      var data = await apiGet('/admin/api.php?action=get&code=' + encodeURIComponent(code));
      if (data && data.ok === true && data.row) {
        await fillForm(data.row);
        window.scrollTo({ top: 0, behavior: 'smooth' });
        toast.info('Editing: ' + code);
      } else {
        toast.error((data && data.error) ? data.error : 'Failed to load shortlink');
      }
      return;
    }

    if (act === 'delete') {
      var confirmed = await customConfirm('Delete shortlink "' + code + '"?\n\nThis action cannot be undone.');
      if (!confirmed) return;

      t.setAttribute('disabled', 'disabled');
      t.textContent = 'Deleting...';

      var r = await apiPost('delete', { code: code });
      if (r && r.ok === true) {
        toast.success('Shortlink deleted: ' + code);
        await load();
      } else {
        toast.error((r && r.error) ? r.error : 'Failed to delete shortlink');
        t.removeAttribute('disabled');
        t.textContent = 'Delete';
      }
    }
  }

  if (tblEl) tblEl.addEventListener('click', onTableClick);

  async function loadAvailableDomains() {
    var data = await apiGet('/admin/api.php?action=list_available_domains');
    var domainSelector = byId('domain_selector');

    if (!domainSelector) return;

    if (!data || data.ok !== true) {
      domainSelector.innerHTML = '<option value="">No domains available</option>';
      return;
    }

    var globalDomains = data.global_domains || [];
    var userDomains = data.user_domains || [];

    availableDomains = [];
    for (var i = 0; i < globalDomains.length; i++) {
      availableDomains.push({ type: 'global', id: globalDomains[i].id, domain: globalDomains[i].domain });
    }
    for (var j = 0; j < userDomains.length; j++) {
      availableDomains.push({ type: 'user', id: userDomains[j].id, domain: userDomains[j].domain });
    }

    var options = '<option value="">Use default domain</option>';

    if (availableDomains.length > 0) {
      options += '<option value="random" class="option-highlight">🎲 Random Domain - Acak dari ' + availableDomains.length + ' domain</option>';
    }

    if (globalDomains.length > 0) {
      options += '<optgroup label="Global Domains (Available to all users)">';
      for (var k = 0; k < globalDomains.length; k++) {
        var d1 = globalDomains[k];
        var defaultLabel1 = d1.is_default === 1 ? ' (Default)' : '';
        options += '<option value="global:' + String(d1.id) + '">' + esc(d1.domain) + defaultLabel1 + '</option>';
      }
      options += '</optgroup>';
    }

    if (userDomains.length > 0) {
      options += '<optgroup label="My Personal Domains">';
      for (var m = 0; m < userDomains.length; m++) {
        var d2 = userDomains[m];
        var defaultLabel2 = d2.is_default === 1 ? ' (Default)' : '';
        options += '<option value="user:' + String(d2.id) + '">' + esc(d2.domain) + defaultLabel2 + '</option>';
      }
      options += '</optgroup>';
    }

    domainSelector.innerHTML = options;
  }

  if (frm) {
    frm.addEventListener('submit', async function (ev) {
      ev.preventDefault();

      var submitBtn = byId('submit');
      var bulkCountInput = byId('bulk_count');
      var bulkOutputEl = byId('bulk_output');

      if (!submitBtn || !bulkCountInput || !bulkOutputEl) return;

      var bulkCount = parseInt(bulkCountInput.value, 10) || 1;
      if (bulkCount < 1 || bulkCount > 25) {
        toast.error('Jumlah shortlink harus antara 1-25');
        return;
      }

      var codeVal = (byId('code').value || '').trim();
      if (!validateCode(codeVal, true)) {
        toast.error('Custom Code tidak valid. Pakai 3-64 char: huruf/angka/underscore/dash.');
        return;
      }

      var ogTitle = (byId('og_title').value || '').trim();
      var ogDesc = (byId('og_description').value || '').trim();
      var ogImg = (byId('og_image_url').value || '').trim();

      var ogCheck = validateOg(ogTitle, ogDesc, ogImg);
      if (!ogCheck.ok) {
        toast.error(ogCheck.error || 'OG validation failed');
        return;
      }

      var domainSelectorValue = (byId('domain_selector').value || '').trim();
      var domainType = null;
      var domainId = null;

      if (domainSelectorValue === 'random') {
        if (availableDomains.length > 0) {
          var ri = Math.floor(Math.random() * availableDomains.length);
          domainType = availableDomains[ri].type;
          domainId = availableDomains[ri].id;
          toast.info('🎲 Random domain dipilih: ' + availableDomains[ri].domain);
        } else {
          toast.error('Tidak ada domain tersedia untuk random selection');
          return;
        }
      } else if (domainSelectorValue !== '') {
        var parts = domainSelectorValue.split(':');
        if (parts.length === 2) {
          domainType = parts[0];
          domainId = parseInt(parts[1], 10) || null;
        }
      }

      var basePayload = {
        code: codeVal,
        og_title: ogTitle,
        og_description: ogDesc,
        og_image_url: ogImg,
        og_type: (byId('og_type').value || '').trim(),
        og_site_name: (byId('og_site_name').value || '').trim(),
        og_locale: (byId('og_locale').value || '').trim(),
        use_shim: byId('use_shim').checked ? 1 : 0,
        domain_type: domainType,
        domain_id: domainId
      };

      submitBtn.disabled = true;
      var originalText = submitBtn.textContent;
      submitBtn.textContent = editingCode ? 'Updating...' : 'Creating...';
      bulkOutputEl.value = '';

      try {
        if (editingCode) {
          basePayload.code = editingCode;

          var upd = await apiPost('update', basePayload);
          if (upd && upd.ok === true) {
            toast.success('Shortlink updated: ' + editingCode);
            await setFormModeCreate();
            await load();
            return;
          }

          toast.error((upd && upd.error) ? upd.error : 'Update gagal');
          return;
        }

        var results = [];
        var successCount = 0;
        var failCount = 0;
        var usedDomains = {};

        var thirdPartyService = byId('third_party_service').value || '';

        for (var i = 0; i < bulkCount; i++) {
          var payload = JSON.parse(JSON.stringify(basePayload));

          if (domainSelectorValue === 'random' && availableDomains.length > 0) {
            var idx = Math.floor(Math.random() * availableDomains.length);
            payload.domain_type = availableDomains[idx].type;
            payload.domain_id = availableDomains[idx].id;
            usedDomains[availableDomains[idx].domain] = true;
          }

          if (payload.code !== '' && bulkCount > 1) {
            payload.code = payload.code + '_' + String(i + 1);
          }

          var r = await apiPost('create', payload);

          if (r && r.ok === true) {
            var code = r.code || payload.code;
            var fullUrl = '';

            if (payload.domain_type && payload.domain_id) {
              var picked = null;
              for (var p = 0; p < availableDomains.length; p++) {
                if (availableDomains[p].type === payload.domain_type && availableDomains[p].id === payload.domain_id) {
                  picked = availableDomains[p];
                  break;
                }
              }
              fullUrl = picked ? ('https://' + picked.domain + '/' + code) : (shortUrl(code));
            } else {
              fullUrl = shortUrl(code);
            }

            if (thirdPartyService) {
              submitBtn.textContent = 'Creating 3rd party link ' + String(i + 1) + '/' + String(bulkCount) + '...';
              var tp = await createThirdPartyShortlink(thirdPartyService, fullUrl);

              if (tp.ok) {
                results.push(tp.url + ' → ' + fullUrl);
                successCount++;
              } else {
                results.push('ERROR (3rd party): ' + (tp.error || 'Unknown') + ' | Internal: ' + fullUrl);
                failCount++;
              }
            } else {
              results.push(fullUrl);
              successCount++;
            }
          } else {
            results.push('ERROR: ' + ((r && r.error) ? r.error : 'Unknown error'));
            failCount++;
          }
        }

        bulkOutputEl.value = results.join('\n');

        var usedDomainCount = Object.keys(usedDomains).length;

        if (failCount === 0) {
          var msgOk = 'Successfully created ' + String(successCount) + ' shortlink(s)';
          if (domainSelectorValue === 'random' && usedDomainCount > 0) {
            msgOk += ' using ' + String(usedDomainCount) + ' unique domain(s)';
          }
          toast.success(msgOk);
          await load();
          await setFormModeCreate();
        } else {
          var msgErr = 'Created ' + String(successCount) + ', Failed ' + String(failCount);
          if (domainSelectorValue === 'random' && usedDomainCount > 0) {
            msgErr += ' (used ' + String(usedDomainCount) + ' domain(s))';
          }
          toast.error(msgErr);
          await load();
        }
      } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    });
  }

  var prevBtn = byId('prev');
  if (prevBtn) {
    prevBtn.addEventListener('click', function () {
      if (page > 1) {
        page--;
        load();
      }
    });
  }

  var nextBtn = byId('next');
  if (nextBtn) {
    nextBtn.addEventListener('click', function () {
      page++;
      load();
    });
  }

  var resetBtn = byId('reset');
  if (resetBtn) resetBtn.addEventListener('click', setFormModeCreate);

  var copyBtn = byId('copyOutputBtn');
  if (copyBtn) {
    copyBtn.addEventListener('click', async function () {
      var outputEl = byId('bulk_output');
      if (!outputEl) return;

      var textToCopy = (outputEl.value || '').trim();
      if (textToCopy === '') {
        toast.error('No output to copy');
        return;
      }

      try {
        if (navigator.clipboard && navigator.clipboard.writeText) {
          await navigator.clipboard.writeText(textToCopy);
        } else {
          outputEl.select();
          document.execCommand('copy');
        }

        toast.success('Copied ' + String(textToCopy.split('\n').length) + ' link(s) to clipboard!');

        var btn = this;
        var old = btn.textContent;
        btn.textContent = '✓ Copied!';
        btn.classList.add('btn-copy-success');

        setTimeout(function () {
          btn.textContent = old;
          btn.classList.remove('btn-copy-success');
        }, 2000);
      } catch (e) {
        toast.error('Copy failed');
      }
    });
  }

  // Init
  setFormModeCreate();
  loadAvailableDomains();
  load();
})();
</script>
</body>
</html>
