<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Db;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';
\App\Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::initSession();
$csrf = Auth::ensureCsrfToken();

$error = '';
$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
}

function jsonResponse(bool $ok, string $error = '', string $redirect = ''): void
{
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'ok' => $ok,
        'error' => $error,
        'redirect' => $redirect
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

function isSafeConfigError(Throwable $e): bool
{
    $msg = $e->getMessage();
    return str_starts_with($msg, 'CONFIG:');
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
        $token = (string) ($_POST['csrf_token'] ?? '');

        if (!Auth::verifyCsrfToken($token)) {
            $error = 'CSRF token invalid.';
            if ($isAjax) {
                jsonResponse(false, $error);
            }
        } else {
            $username = trim((string) ($_POST['username'] ?? ''));
            $password = trim((string) ($_POST['password'] ?? ''));

            $pdo = Db::pdo();

            if ($username !== '' && $password !== '' && Auth::login($pdo, $username, $password)) {
                if ($isAjax) {
                    jsonResponse(true, '', '/admin/index.php');
                }
                header('Location: /admin/index.php', true, 302);
                exit;
            }

            $error = 'Username atau password salah.';
            if ($isAjax) {
                jsonResponse(false, $error);
            }
        }
    }
} catch (Throwable $e) {
    $rid = bin2hex(random_bytes(8));
    error_log('admin_login_error rid=' . $rid . ' type=' . get_class($e) . ' msg=' . $e->getMessage());

    header('X-Request-ID: ' . $rid);

    if (isSafeConfigError($e)) {
        $error = trim(substr($e->getMessage(), 7));
    } else {
        $error = 'Server error. ID: ' . $rid;
    }

    if ($isAjax) {
        jsonResponse(false, $error);
    }
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <title>Admin Login - OG Shortlink Dashboard</title>
  <link rel="stylesheet" href="/admin/assets/login.css" nonce="<?= e($nonce) ?>">
  <link rel="stylesheet" href="/admin/assets/toast.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="wrap">
    <div class="card">
      <div class="card-header">
        <h1 class="title">Admin Login</h1>
        <p class="subtitle">OG Shortlink Dashboard</p>
      </div>

      <form method="post" action="/admin/login.php" autocomplete="off" id="loginForm">
        <input type="hidden" name="csrf_token" value="<?= e($csrf) ?>">

        <div class="field">
          <label for="username" class="label-required">Username</label>
          <input type="text" id="username" name="username" required autofocus>
        </div>

        <div class="field">
          <label for="password" class="label-required">Password</label>
          <input type="password" id="password" name="password" required>
        </div>

        <button class="btn btn-primary" type="submit" id="submitBtn">Masuk</button>
      </form>

      <div class="footer">
        <p>&copy; <?= date('Y') ?> OG Shortlink</p>
        <span class="badge-env"><?= getenv('APP_ENV') ?: 'production' ?></span>
      </div>
    </div>
  </div>

  <script src="/admin/assets/toast.js" nonce="<?= e($nonce) ?>"></script>
  <script nonce="<?= e($nonce) ?>">
  (function(){
    'use strict';

    <?php if ($error !== '') : ?>
    // Show initial error from server
    window.addEventListener('DOMContentLoaded', function() {
      toast.error(<?= json_encode($error, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) ?>);
    });
    <?php endif; ?>

    const loginForm = document.getElementById('loginForm');
    const submitBtn = document.getElementById('submitBtn');

    if (loginForm && submitBtn) {
      loginForm.addEventListener('submit', async function(e) {
        e.preventDefault();

        // Disable submit
        submitBtn.disabled = true;
        submitBtn.textContent = 'Logging in...';

        const formData = new FormData(loginForm);

        try {
          const response = await fetch('/admin/login.php', {
            method: 'POST',
            headers: {
              'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData,
            credentials: 'same-origin'
          });

          const contentType = response.headers.get('content-type');

          if (contentType && contentType.includes('application/json')) {
            const data = await response.json();

            if (data.ok) {
              toast.success('Login berhasil!', 1000);
              submitBtn.textContent = 'Success!';
              setTimeout(function() {
                window.location.href = data.redirect || '/admin/index.php';
              }, 300);
            } else {
              toast.error(data.error || 'Login gagal');
              submitBtn.disabled = false;
              submitBtn.textContent = 'Masuk';
            }
          } else {
            window.location.href = '/admin/index.php';
          }
        } catch (error) {
          toast.error('Network error. Please try again.');
          submitBtn.disabled = false;
          submitBtn.textContent = 'Masuk';
        }
      });
    }
  })();
  </script>
</body>
</html>
