<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';

\App\Env::bootstrap();

$nonce = Http::nonce();

Auth::requireAuth();
Auth::requireAdmin();

$csrf = Auth::ensureCsrfToken();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

const DEFAULTS_FILE = __DIR__ . '/../../var/admin-settings/defaults.json';

/**
 * @return array<string, string>
 */
function defaultsSchema(): array
{
    return [
        'default_site_name' => '',
        'default_locale' => 'en_US',
        'default_og_type' => 'website',
        'redirect_url' => '',
    ];
}

/**
 * @return array<string, string>
 */
function loadDefaults(): array
{
    $default = defaultsSchema();
    $path = DEFAULTS_FILE;

    $dir = dirname($path);
    if (!is_dir($dir)) {
        @mkdir($dir, 0750, true);
    }

    if (!is_file($path)) {
        return $default;
    }

    $raw = (string)file_get_contents($path);
    if ($raw === '') {
        return $default;
    }

    try {
        $decoded = json_decode($raw, true, 512, JSON_THROW_ON_ERROR);
    } catch (Throwable $e) {
        return $default;
    }

    if (!is_array($decoded)) {
        return $default;
    }

    /** @var array<string, mixed> $decoded */
    $merged = array_merge($default, $decoded);

    return [
        'default_site_name' => (string)($merged['default_site_name'] ?? ''),
        'default_locale' => (string)($merged['default_locale'] ?? 'en_US'),
        'default_og_type' => (string)($merged['default_og_type'] ?? 'website'),
        'redirect_url' => (string)($merged['redirect_url'] ?? ''),
    ];
}

/**
 * @param array<string, string> $data
 */
function saveDefaults(array $data): bool
{
    $path = DEFAULTS_FILE;
    $dir = dirname($path);

    if (!is_dir($dir) && !mkdir($dir, 0750, true) && !is_dir($dir)) {
        return false;
    }

    $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    if ($json === false) {
        return false;
    }

    $tmp = $path . '.tmp.' . bin2hex(random_bytes(6));
    $ok = file_put_contents($tmp, $json, LOCK_EX);
    if ($ok === false) {
        @unlink($tmp);
        return false;
    }

    @chmod($tmp, 0640);

    if (!rename($tmp, $path)) {
        @unlink($tmp);
        return false;
    }

    return true;
}

function isAllowedLocale(string $v): bool
{
    $allowed = ['en_US', 'id_ID', 'en_GB', 'es_ES', 'fr_FR', 'de_DE', 'ja_JP', 'zh_CN'];
    return in_array($v, $allowed, true);
}

function isAllowedOgType(string $v): bool
{
    $allowed = ['website', 'article', 'product', 'video', 'music', 'book', 'profile'];
    return in_array($v, $allowed, true);
}

function normalizeUrl(string $url): string
{
    $url = trim($url);
    if ($url === '') {
        return '';
    }
    if (strlen($url) > 2048) {
        return '';
    }
    $ok = filter_var($url, FILTER_VALIDATE_URL);
    if ($ok === false) {
        return '';
    }
    $parts = parse_url($url);
    if (!is_array($parts)) {
        return '';
    }
    $scheme = strtolower((string)($parts['scheme'] ?? ''));
    if ($scheme !== 'https' && $scheme !== 'http') {
        return '';
    }
    return $url;
}

$settings = loadDefaults();
$success = '';
$error = '';
$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

/**
 * Send JSON response for AJAX requests
 */
function jsonResponse(bool $ok, string $error = '', string $message = ''): void
{
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'ok' => $ok,
        'error' => $error,
        'message' => $message
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
    $token = (string)($_POST['csrf_token'] ?? '');

    if (!Auth::verifyCsrfToken($token)) {
        $error = 'CSRF token invalid.';
        if ($isAjax) {
            jsonResponse(false, $error);
        }
    } else {
        $siteName = trim((string)($_POST['site_name'] ?? ''));
        if (strlen($siteName) > 120) {
            $siteName = substr($siteName, 0, 120);
        }

        $locale = trim((string)($_POST['locale'] ?? 'en_US'));
        if (!isAllowedLocale($locale)) {
            $locale = 'en_US';
        }

        $ogType = trim((string)($_POST['og_type'] ?? 'website'));
        if (!isAllowedOgType($ogType)) {
            $ogType = 'website';
        }

        $redirectUrl = normalizeUrl((string)($_POST['redirect_url'] ?? ''));
        if ($redirectUrl === '') {
            $error = 'Redirect URL invalid.';
            if ($isAjax) {
                jsonResponse(false, $error);
            }
        } else {
            $newSettings = [
                'default_site_name' => $siteName,
                'default_locale' => $locale,
                'default_og_type' => $ogType,
                'redirect_url' => $redirectUrl,
            ];

            if (saveDefaults($newSettings)) {
                $settings = $newSettings;
                $success = 'Settings saved successfully!';
                if ($isAjax) {
                    jsonResponse(true, '', $success);
                }
            } else {
                $error = 'Failed to save settings.';
                if ($isAjax) {
                    jsonResponse(false, $error);
                }
            }
        }
    }
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <title>Settings | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Settings</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <a href="/admin/users.php">Users</a>
        <a href="/admin/domains.php">Domains</a>
        <a href="/admin/global-domains.php">Global Domains</a>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php" aria-current="page">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">Settings</h1>
            <div class="page-sub">Configure application, security, and protection settings</div>
          </div>
        </div>

    <div class="tabs">
      <div class="tab active" data-tab="defaults">Defaults</div>
      <div class="tab" data-tab="security">Security</div>
      <div class="tab" data-tab="facebook">FB Protection</div>
      <div class="tab" data-tab="shim">Shim Config</div>
      <div class="tab" data-tab="monitoring">Monitoring</div>
    </div>

        <div id="tab-defaults" class="tab-content active">
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Default Settings</h3>
                <div class="card-sub">Set default values for new shortlinks</div>
              </div>
            </div>

            <?php if ($error !== '') : ?>
              <div class="alert alert-error auto-hide-alert" style="margin-top: 12px;"><?= e($error) ?></div>
            <?php endif; ?>

            <?php if ($success !== '') : ?>
              <div class="alert alert-success auto-hide-alert" style="margin-top: 12px;"><?= e($success) ?></div>
            <?php endif; ?>

            <div id="defaultsError" class="alert alert-error" style="display:none;"></div>
            <div id="defaultsSuccess" class="alert alert-success" style="display:none;"></div>

            <form id="defaultsForm" method="post" action="/admin/settings.php" autocomplete="off">
              <input type="hidden" name="csrf_token" value="<?= e($csrf) ?>">

              <div>
                <label for="site_name">Default Site Name</label>
                <input type="text" id="site_name" name="site_name"
                       value="<?= e($settings['default_site_name']) ?>"
                       placeholder="e.g., My Brand">
              </div>

              <div style="margin-top: 12px;">
                <label for="locale">Default Locale</label>
                <select id="locale" name="locale" required>
                  <option value="en_US" <?= $settings['default_locale'] === 'en_US' ? 'selected' : '' ?>>English (US)</option>
                  <option value="id_ID" <?= $settings['default_locale'] === 'id_ID' ? 'selected' : '' ?>>Indonesian</option>
                  <option value="en_GB" <?= $settings['default_locale'] === 'en_GB' ? 'selected' : '' ?>>English (UK)</option>
                  <option value="es_ES" <?= $settings['default_locale'] === 'es_ES' ? 'selected' : '' ?>>Spanish</option>
                  <option value="fr_FR" <?= $settings['default_locale'] === 'fr_FR' ? 'selected' : '' ?>>French</option>
                  <option value="de_DE" <?= $settings['default_locale'] === 'de_DE' ? 'selected' : '' ?>>German</option>
                  <option value="ja_JP" <?= $settings['default_locale'] === 'ja_JP' ? 'selected' : '' ?>>Japanese</option>
                  <option value="zh_CN" <?= $settings['default_locale'] === 'zh_CN' ? 'selected' : '' ?>>Chinese (Simplified)</option>
                </select>
              </div>

              <div style="margin-top: 12px;">
                <label for="og_type">Default OG Type</label>
                <select id="og_type" name="og_type" required>
                  <option value="website" <?= $settings['default_og_type'] === 'website' ? 'selected' : '' ?>>Website</option>
                  <option value="article" <?= $settings['default_og_type'] === 'article' ? 'selected' : '' ?>>Article</option>
                  <option value="product" <?= $settings['default_og_type'] === 'product' ? 'selected' : '' ?>>Product</option>
                  <option value="video" <?= $settings['default_og_type'] === 'video' ? 'selected' : '' ?>>Video</option>
                  <option value="music" <?= $settings['default_og_type'] === 'music' ? 'selected' : '' ?>>Music</option>
                  <option value="book" <?= $settings['default_og_type'] === 'book' ? 'selected' : '' ?>>Book</option>
                  <option value="profile" <?= $settings['default_og_type'] === 'profile' ? 'selected' : '' ?>>Profile</option>
                </select>
              </div>

              <div style="margin-top: 12px;">
                <label for="redirect_url">Global Redirect URL</label>
                <input type="url" id="redirect_url" name="redirect_url"
                       value="<?= e($settings['redirect_url']) ?>"
                       placeholder="https://example.com/landing"
                       required>
                <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">All shortlinks will redirect to this URL (URL cloaking)</div>
              </div>

              <div style="margin-top: 16px;">
                <button class="btn" type="submit">Save Settings</button>
              </div>
            </form>
          </div>
        </div>

        <div id="tab-security" class="tab-content">
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Security Configuration</h3>
                <div class="card-sub">Configure destination safety, IP intelligence, and signed URLs</div>
              </div>
            </div>

            <div id="securityError" class="alert alert-error" style="display:none;"></div>
            <div id="securitySuccess" class="alert alert-success" style="display:none;"></div>

            <form id="securityFrm" autocomplete="off">
              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Destination Safety</legend>

                <div>
                  <label for="sec_allowed_hosts">Allowed Hosts (comma-separated)</label>
                  <input id="sec_allowed_hosts" name="allowed_hosts" placeholder="example.com,*.example.com,app.example.com">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Kosongkan untuk allow semua. Gunakan *.domain.com untuk wildcard subdomain.</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_denied_hosts">Denied Hosts (comma-separated)</label>
                  <input id="sec_denied_hosts" name="denied_hosts" placeholder="malware.com,spam.net,phishing.org">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Blacklist domains (prioritas tertinggi).</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_denied_tlds">Denied TLDs (comma-separated)</label>
                  <input id="sec_denied_tlds" name="denied_tlds" placeholder="tk,ml,ga,cf,gq,xxx,adult,porn">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Kosongkan untuk pakai default (tk,ml,ga,cf,gq,xxx,adult,porn,zip,mov).</div>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="sec_https_only" name="https_only" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Force HTTPS Only</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Tolak semua URL HTTP, hanya izinkan HTTPS.</div>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="sec_block_ip" name="block_ip_literal" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Block IP Literal Hosts</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Tolak URL dengan IP address (http://192.168.1.1).</div>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="sec_block_userinfo" name="block_userinfo" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Block Userinfo in URLs</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Tolak URL dengan credentials (http://user:pass@host).</div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">IP Intelligence & Risk Assessment</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="sec_ip_intel_enabled" name="ip_intel_enabled" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable IP Intelligence</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Aktifkan risk-based routing (risky IP → safe URL).</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_risk_threshold">Risk Threshold (0-100)</label>
                  <input type="number" id="sec_risk_threshold" name="risk_threshold" min="0" max="100" value="75">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">IP dengan risk score &gt;= threshold dianggap risky.</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_safe_url">Safe URL (for risky IPs)</label>
                  <input type="url" id="sec_safe_url" name="safe_url" placeholder="https://example.com/warning">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Landing page untuk IP yang risky (VPN/Proxy/Tor/Datacenter).</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_fallback_url">Fallback URL (when API fails)</label>
                  <input type="url" id="sec_fallback_url" name="fallback_url" placeholder="https://example.com/maintenance">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">URL ketika IP intel API error/timeout.</div>
                </div>

                <hr style="border: 0; border-top: 1px solid var(--border-color); margin: 20px 0;">
                <div style="font-weight: 600; font-size: 0.85rem; margin-bottom: 12px;">API Configuration</div>

                <div>
                  <label for="sec_blackbox_url">Blackbox API URL</label>
                  <input type="url" id="sec_blackbox_url" name="blackbox_api_url" placeholder="https://blackbox.example.com/check">
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_ipquery_url">IPQuery API URL</label>
                  <input type="url" id="sec_ipquery_url" name="ipquery_api_url" placeholder="https://ipquery.example.com/api/v1/lookup">
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_ipquery_key">IPQuery API Key</label>
                  <input type="password" id="sec_ipquery_key" name="ipquery_api_key" placeholder="(leave blank to keep)">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;" id="sec_ipquery_key_hint"></div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_atlas_url">Atlas API URL (fallback)</label>
                  <input type="url" id="sec_atlas_url" name="atlas_api_url" placeholder="https://atlas.example.com/ip">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Signed URLs (Anti Abuse)</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="sec_sign_enabled" name="sign_enabled" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Signed URLs</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Require HMAC signature + TTL untuk semua shortlinks.</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_sign_secret">URL Signing Secret</label>
                  <input type="password" id="sec_sign_secret" name="sign_secret" placeholder="(leave blank to keep)">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;" id="sec_sign_secret_hint">Min 32 chars. Generate: openssl rand -base64 32</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="sec_sign_ttl">Default TTL (seconds)</label>
                  <input type="number" id="sec_sign_ttl" name="sign_ttl" min="0" value="3600">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">3600 = 1 hour, 86400 = 1 day.</div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">cPanel API (Domain Management)</legend>

                <div>
                  <label for="cpanel_host">cPanel Hostname</label>
                  <input id="cpanel_host" name="cpanel_host" placeholder="cpanel.example.com">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Format: cpanel.yourdomain.com (without https://)</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="cpanel_username">cPanel Username</label>
                  <input id="cpanel_username" name="cpanel_username" placeholder="cpanel_user">
                </div>

                <div style="margin-top: 12px;">
                  <label for="cpanel_token">cPanel API Token</label>
                  <textarea id="cpanel_token" name="cpanel_token" rows="2" placeholder="(leave blank to keep)"></textarea>
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;" id="cpanel_token_hint"></div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Cloudflare API (DNS Management)</legend>

                <div>
                  <label for="cf_api_token">Cloudflare API Token</label>
                  <textarea id="cf_api_token" name="cf_api_token" rows="2" placeholder="(leave blank to keep)"></textarea>
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;" id="cf_api_token_hint">Required permissions: Zone:DNS:Edit</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="cf_default_ttl">Default TTL</label>
                  <input type="number" id="cf_default_ttl" name="cf_default_ttl" min="1" max="86400" value="1">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Time to Live in seconds (1 = Auto, 120-86400 for custom)</div>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="cf_default_proxied" name="cf_default_proxied" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Cloudflare Proxy by default (Orange Cloud)</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">New DNS records will use CDN/protection</div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Rate Limiting</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="rate_limit_enabled" name="rate_limit_enabled" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Rate Limiting</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Protect endpoints from abuse with request throttling</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="rate_limit_max_requests">Max Requests per Window</label>
                  <input type="number" id="rate_limit_max_requests" name="rate_limit_max_requests" min="1" max="10000" value="100" placeholder="100">
                </div>

                <div style="margin-top: 12px;">
                  <label for="rate_limit_window">Time Window (seconds)</label>
                  <input type="number" id="rate_limit_window" name="rate_limit_window" min="1" max="3600" value="60" placeholder="60">
                </div>

                <div style="margin-top: 12px;">
                  <label for="rate_limit_by">Rate Limit By</label>
                  <select id="rate_limit_by" name="rate_limit_by">
                    <option value="ip">IP Address</option>
                    <option value="user">User ID (authenticated users only)</option>
                    <option value="ip_user">IP + User ID (combined)</option>
                  </select>
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Identifier used for rate limiting</div>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="rate_limit_whitelist_admin" name="rate_limit_whitelist_admin" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Whitelist Admin Users</span>
                  </label>
                </div>
              </fieldset>

              <div style="display: flex; gap: 8px; margin-top: 24px;">
                <button class="btn" type="submit">Save Security Settings</button>
                <button class="btn btn-secondary" type="button" id="btnSecurityReload">Reload</button>
              </div>

              <div class="alert alert-warning" style="margin-top: 12px;">
                <strong>Important:</strong> Perubahan security settings akan langsung berlaku. Test sebelum production.
              </div>
            </form>
          </div>
        </div>

        <div id="tab-facebook" class="tab-content">
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Facebook Protection</h3>
                <div class="card-sub">Mini WAF untuk melindungi domain dari ban</div>
              </div>
            </div>

            <div id="fbError" class="alert alert-error" style="display:none;"></div>
            <div id="fbSuccess" class="alert alert-success" style="display:none;"></div>

            <form id="facebookFrm" autocomplete="off">
              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Protection Status</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_enabled" name="fb_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Facebook Protection</span>
                  </label>
                  <div class="muted" style="margin-top: 4px; margin-left: 24px; font-size: 0.78rem;">Aktifkan protection untuk prevent domain ban</div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Rate Limiting (per Domain)</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_rate_enabled" name="fb_rate_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Rate Limiting</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_rate_window">Time Window (seconds)</label>
                  <input type="number" id="fb_rate_window" name="fb_rate_window" min="10" max="300" value="60" placeholder="60">
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_rate_max">Max Requests per Domain</label>
                  <input type="number" id="fb_rate_max" name="fb_rate_max" min="10" max="200" value="30" placeholder="30">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Default: 30</div>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_rate_suspicious">Suspicious Threshold (req/sec)</label>
                  <input type="number" id="fb_rate_suspicious" name="fb_rate_suspicious" min="5" max="50" value="10" placeholder="10">
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_rate_cooldown">Violation Cooldown (seconds)</label>
                  <input type="number" id="fb_rate_cooldown" name="fb_rate_cooldown" min="60" max="3600" value="600" placeholder="600">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Destination Validation</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_dest_enabled" name="fb_dest_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Destination Validation</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_dest_block_tlds" name="fb_dest_block_tlds" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Block Suspicious TLDs</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_dest_tlds">Suspicious TLDs (comma-separated)</label>
                  <input id="fb_dest_tlds" name="fb_dest_tlds" value=".tk,.ml,.ga,.cf,.gq,.xyz,.top" placeholder=".tk,.ml,.ga">
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_dest_scan_path" name="fb_dest_scan_path" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Scan Path for Malicious Keywords</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_dest_keywords">Malicious Keywords (comma-separated)</label>
                  <input id="fb_dest_keywords" name="fb_dest_keywords" value="phishing,scam,fraud,hack,crack,warez,malware" placeholder="phishing,scam,fraud">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">IP Verification</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_ip_enabled" name="fb_ip_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable IP Verification</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_ip_strict" name="fb_ip_strict" value="1" style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Strict Mode (Reject non-FB IPs)</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_ip_log_suspicious" name="fb_ip_log_suspicious" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Log Suspicious Requests</span>
                  </label>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Domain Health Monitoring</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_health_enabled" name="fb_health_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Health Monitoring</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_health_cache_ttl">Cache TTL (seconds)</label>
                  <input type="number" id="fb_health_cache_ttl" name="fb_health_cache_ttl" min="60" max="3600" value="300" placeholder="300">
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_health_ban_duration">Ban Duration (seconds)</label>
                  <input type="number" id="fb_health_ban_duration" name="fb_health_ban_duration" min="300" max="86400" value="3600" placeholder="3600">
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_health_cooldown_duration">Cooldown Duration (seconds)</label>
                  <input type="number" id="fb_health_cooldown_duration" name="fb_health_cooldown_duration" min="60" max="3600" value="600" placeholder="600">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Response Headers Optimization</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_headers_vary" name="fb_headers_vary" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Vary by User-Agent</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_headers_allow_bot_cache" name="fb_headers_allow_bot_cache" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Allow Bot Caching</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label for="fb_headers_bot_cache_ttl">Bot Cache TTL (seconds)</label>
                  <input type="number" id="fb_headers_bot_cache_ttl" name="fb_headers_bot_cache_ttl" min="60" max="3600" value="300" placeholder="300">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Logging & Monitoring</legend>

                <div>
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_log_enabled" name="fb_log_enabled" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Enable Logging</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_log_violations" name="fb_log_violations" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Log Rate Limit Violations</span>
                  </label>
                </div>

                <div style="margin-top: 12px;">
                  <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" id="fb_log_destination_blocks" name="fb_log_destination_blocks" value="1" checked style="width: 16px; height: 16px;">
                    <span style="font-size: 0.82rem;">Log Destination Blocks</span>
                  </label>
                </div>
              </fieldset>

              <div style="display: flex; gap: 8px; margin-top: 24px;">
                <button class="btn" type="submit">Save Facebook Protection Settings</button>
                <button class="btn btn-secondary" type="button" id="btnFbReload">Reload</button>
              </div>

              <div class="alert alert-info" style="margin-top: 12px;">
                <strong>Info:</strong> Facebook Protection menggunakan APCu untuk caching. Pastikan APCu enabled.
              </div>
            </form>
          </div>
        </div>

        <div id="tab-shim" class="tab-content">
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Shim Configuration</h3>
                <div class="card-sub">Configure professional shim link settings (V2)</div>
              </div>
            </div>

            <div id="shimError" class="alert alert-error" style="display:none;"></div>
            <div id="shimSuccess" class="alert alert-success" style="display:none;"></div>

            <form id="shimFrm" autocomplete="off">
              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Security</legend>

                <div>
                  <label for="shim_secret">HMAC Secret</label>
                  <input type="password" id="shim_secret" name="shim_secret" placeholder="(leave blank to keep)">
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;">Generate: <code style="font-size: 0.78rem;">openssl rand -hex 32</code></div>
                  <div class="muted" style="margin-top: 4px; font-size: 0.78rem;" id="shim_secret_hint"></div>
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Destination Filtering</legend>

                <div>
                  <label for="shim_allowed_hosts">Allowed Hosts (comma-separated)</label>
                  <input id="shim_allowed_hosts" name="shim_allowed_hosts" placeholder="example.com,app.example.com,*.trusted.com">
                </div>

                <div style="margin-top: 12px;">
                  <label for="shim_allowed_suffixes">Allowed Suffixes (comma-separated)</label>
                  <input id="shim_allowed_suffixes" name="shim_allowed_suffixes" placeholder=".example.com,.trusted.com">
                </div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Shim Version</legend>

                <div>
                  <label for="shim_version">Active Shim Version</label>
                  <select id="shim_version" name="shim_version">
                    <option value="v1">V1 - Classic (shim.php)</option>
                    <option value="v2" selected>V2 - Professional (shim_v2.php)</option>
                  </select>
                </div>
              </fieldset>

              <div style="display: flex; gap: 8px; margin-top: 24px;">
                <button class="btn" type="submit">Save Shim Settings</button>
                <button class="btn btn-secondary" type="button" id="btnShimReload">Reload</button>
              </div>

              <div class="alert alert-warning" style="margin-top: 12px;">
                <strong>Important:</strong> SHIM_HMAC_SECRET minimal 32 chars. Rotate berkala.
              </div>
            </form>
          </div>
        </div>

        <div id="tab-monitoring" class="tab-content">
          <div class="card">
            <div class="card-head">
              <div>
                <h3 class="card-title">Domain Health Monitoring</h3>
                <div class="card-sub">Real-time monitoring untuk Facebook Protection dan Shim metrics</div>
              </div>
            </div>

            <div style="margin-top: 16px;">
              <button class="btn btn-secondary" type="button" id="btnRefreshMetrics">Refresh Metrics</button>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-top: 16px; margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Facebook Protection Metrics</legend>
                <div id="fbMetrics" style="font-family: monospace; font-size: 0.85rem;">Loading metrics...</div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Shim V2 Metrics (APCu)</legend>
                <div id="shimMetrics" style="font-family: monospace; font-size: 0.85rem;">Loading metrics...</div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Domain Health Status</legend>

                <div>
                  <label for="monitor_domain">Domain to Check</label>
                  <div style="display: flex; gap: 8px; margin-top: 6px;">
                    <input type="text" id="monitor_domain" placeholder="example.com" style="flex: 1;">
                    <button class="btn btn-secondary" type="button" id="btnCheckDomain">Check</button>
                  </div>
                </div>

                <div id="domainHealth" style="font-family: monospace; font-size: 0.85rem; margin-top: 12px;">Enter domain and click Check</div>
              </fieldset>

              <fieldset style="border: 1px solid var(--border-color); padding: 16px; border-radius: var(--radius); margin-bottom: 24px;">
                <legend style="font-weight: 600; font-size: 0.9rem; padding: 0 8px;">Recent Protection Events</legend>
                <div id="recentLogs" style="font-family: monospace; font-size: 0.85rem; max-height: 300px; overflow-y: auto;">No recent events</div>
              </fieldset>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

<script nonce="<?= e($nonce) ?>">
(function () {
  'use strict';

  const csrfMeta = document.querySelector('meta[name="csrf-token"]');
  const csrf = csrfMeta ? (csrfMeta.getAttribute('content') || '') : '';

  const tabs = document.querySelectorAll('.tab');
  const tabContents = document.querySelectorAll('.tab-content');
  const tabsContainer = document.querySelector('.tabs');

  tabs.forEach(function (tab) {
    tab.addEventListener('click', function () {
      const targetTab = this.getAttribute('data-tab');
      const targetContent = document.getElementById('tab-' + targetTab);
      if (!targetContent) {
        return;
      }

      tabs.forEach(function (t) { t.classList.remove('active'); });
      tabContents.forEach(function (tc) { tc.classList.remove('active'); });

      this.classList.add('active');
      targetContent.classList.add('active');

      if (tabsContainer) {
        const tabRect = this.getBoundingClientRect();
        const containerRect = tabsContainer.getBoundingClientRect();
        if (tabRect.left < containerRect.left || tabRect.right > containerRect.right) {
          this.scrollIntoView({ behavior: 'smooth', block: 'nearest', inline: 'center' });
        }
      }

      window.scrollTo({ top: 0, behavior: 'smooth' });
    });
  });

  async function apiGet(url) {
    const res = await fetch(url, { credentials: 'same-origin', headers: { 'Accept': 'application/json' } });
    const data = await res.json();
    return data;
  }

  async function apiPost(action, payload) {
    const res = await fetch('/admin/api.php?action=' + encodeURIComponent(action), {
      method: 'POST',
      credentials: 'same-origin',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'X-CSRF-Token': csrf
      },
      body: JSON.stringify(payload || {})
    });

    const data = await res.json();
    return data;
  }

  // =========================
  // Security Configuration
  // =========================

  const securityFrm = document.getElementById('securityFrm');
  const securityError = document.getElementById('securityError');
  const securitySuccess = document.getElementById('securitySuccess');

  function showSecError(msg) {
    securityError.style.display = 'block';
    securityError.textContent = msg;
    securitySuccess.style.display = 'none';
  }

  function showSecSuccess(msg) {
    securitySuccess.style.display = 'block';
    securitySuccess.textContent = msg;
    securityError.style.display = 'none';
    setTimeout(function () {
      securitySuccess.style.transition = 'opacity 0.5s ease';
      securitySuccess.style.opacity = '0';
      setTimeout(function () {
        securitySuccess.style.display = 'none';
        securitySuccess.style.opacity = '1';
      }, 500);
    }, 8000);
  }

  function clearSecMessages() {
    securityError.style.display = 'none';
    securitySuccess.style.display = 'none';
  }

  function fillSecurityForm(config) {
    document.getElementById('sec_allowed_hosts').value = (config.allowed_hosts || []).join(',');
    document.getElementById('sec_denied_hosts').value = (config.denied_hosts || []).join(',');
    document.getElementById('sec_denied_tlds').value = (config.denied_tlds || []).join(',');
    document.getElementById('sec_https_only').checked = config.https_only === true;
    document.getElementById('sec_block_ip').checked = config.block_ip_literal === true;
    document.getElementById('sec_block_userinfo').checked = config.block_userinfo === true;

    document.getElementById('sec_ip_intel_enabled').checked = config.ip_intel_enabled === true;
    document.getElementById('sec_risk_threshold').value = config.risk_threshold || 75;
    document.getElementById('sec_safe_url').value = config.safe_url || '';
    document.getElementById('sec_fallback_url').value = config.fallback_url || '';
    document.getElementById('sec_blackbox_url').value = config.blackbox_api_url || '';
    document.getElementById('sec_ipquery_url').value = config.ipquery_api_url || '';
    document.getElementById('sec_atlas_url').value = config.atlas_api_url || '';

    document.getElementById('sec_ipquery_key').value = '';
    document.getElementById('sec_ipquery_key_hint').textContent = config.ipquery_api_key_set ? 'API key configured (not shown)' : 'API key not set';

    document.getElementById('sec_sign_enabled').checked = config.sign_enabled === true;
    document.getElementById('sec_sign_secret').value = '';
    document.getElementById('sec_sign_secret_hint').textContent = config.sign_secret_set ? 'Signing secret configured (not shown)' : 'Min 32 chars. Generate: openssl rand -base64 32';
    document.getElementById('sec_sign_ttl').value = config.sign_ttl || 3600;

    document.getElementById('cpanel_host').value = config.cpanel_host || '';
    document.getElementById('cpanel_username').value = config.cpanel_username || '';
    document.getElementById('cpanel_token').value = '';
    document.getElementById('cpanel_token_hint').textContent = config.cpanel_token_set ? 'Token configured (not shown)' : 'Token not set';

    document.getElementById('cf_api_token').value = '';
    document.getElementById('cf_api_token_hint').textContent = config.cf_api_token_set ? 'Token configured (not shown)' : 'Required permissions: Zone:DNS:Edit';
    document.getElementById('cf_default_ttl').value = config.cf_default_ttl || 1;
    document.getElementById('cf_default_proxied').checked = config.cf_default_proxied === true;

    document.getElementById('rate_limit_enabled').checked = config.rate_limit_enabled === true;
    document.getElementById('rate_limit_max_requests').value = config.rate_limit_max_requests || 100;
    document.getElementById('rate_limit_window').value = config.rate_limit_window || 60;
    document.getElementById('rate_limit_by').value = config.rate_limit_by || 'ip';
    document.getElementById('rate_limit_whitelist_admin').checked = config.rate_limit_whitelist_admin !== false;
  }

  async function loadSecuritySettings() {
    clearSecMessages();
    const data = await apiGet('/admin/api.php?action=get_security_settings');
    if (data && data.ok === true) {
      fillSecurityForm(data.config || {});
    } else {
      showSecError('Failed to load security settings');
    }
  }

  if (securityFrm) {
    securityFrm.addEventListener('submit', async function (ev) {
      ev.preventDefault();
      clearSecMessages();

      const payload = {
        allowed_hosts: (document.getElementById('sec_allowed_hosts').value || '').trim(),
        denied_hosts: (document.getElementById('sec_denied_hosts').value || '').trim(),
        denied_tlds: (document.getElementById('sec_denied_tlds').value || '').trim(),
        https_only: document.getElementById('sec_https_only').checked,
        block_ip_literal: document.getElementById('sec_block_ip').checked,
        block_userinfo: document.getElementById('sec_block_userinfo').checked,

        ip_intel_enabled: document.getElementById('sec_ip_intel_enabled').checked,
        risk_threshold: parseInt(document.getElementById('sec_risk_threshold').value, 10) || 75,
        safe_url: (document.getElementById('sec_safe_url').value || '').trim(),
        fallback_url: (document.getElementById('sec_fallback_url').value || '').trim(),
        blackbox_api_url: (document.getElementById('sec_blackbox_url').value || '').trim(),
        ipquery_api_url: (document.getElementById('sec_ipquery_url').value || '').trim(),
        ipquery_api_key: (document.getElementById('sec_ipquery_key').value || '').trim(),
        atlas_api_url: (document.getElementById('sec_atlas_url').value || '').trim(),

        sign_enabled: document.getElementById('sec_sign_enabled').checked,
        sign_secret: (document.getElementById('sec_sign_secret').value || '').trim(),
        sign_ttl: parseInt(document.getElementById('sec_sign_ttl').value, 10) || 3600,

        cpanel_host: (document.getElementById('cpanel_host').value || '').trim(),
        cpanel_username: (document.getElementById('cpanel_username').value || '').trim(),
        cpanel_token: (document.getElementById('cpanel_token').value || '').trim(),

        cf_api_token: (document.getElementById('cf_api_token').value || '').trim(),
        cf_default_ttl: parseInt(document.getElementById('cf_default_ttl').value, 10) || 1,
        cf_default_proxied: document.getElementById('cf_default_proxied').checked,

        rate_limit_enabled: document.getElementById('rate_limit_enabled').checked,
        rate_limit_max_requests: parseInt(document.getElementById('rate_limit_max_requests').value, 10) || 100,
        rate_limit_window: parseInt(document.getElementById('rate_limit_window').value, 10) || 60,
        rate_limit_by: document.getElementById('rate_limit_by').value,
        rate_limit_whitelist_admin: document.getElementById('rate_limit_whitelist_admin').checked
      };

      const r = await apiPost('update_security_settings', payload);
      if (r && r.ok === true) {
        showSecSuccess('Security settings saved successfully!');
        window.scrollTo(0, 0);
        await loadSecuritySettings();
      } else {
        showSecError((r && r.error) ? r.error : 'Failed to save settings');
      }
    });
  }

  const btnSecurityReload = document.getElementById('btnSecurityReload');
  if (btnSecurityReload) {
    btnSecurityReload.addEventListener('click', function () {
      loadSecuritySettings();
    });
  }

  const securityTab = document.querySelector('.tab[data-tab="security"]');
  let securityLoaded = false;
  if (securityTab) {
    securityTab.addEventListener('click', function () {
      if (!securityLoaded) {
        loadSecuritySettings();
        securityLoaded = true;
      }
    });
  }

  document.querySelectorAll('.auto-hide-alert').forEach(function (el) {
    setTimeout(function () {
      el.style.transition = 'opacity 0.5s ease';
      el.style.opacity = '0';
      setTimeout(function () { el.remove(); }, 500);
    }, 8000);
  });

  // =========================
  // Defaults Form (AJAX)
  // =========================
  const defaultsForm = document.getElementById('defaultsForm');
  const defaultsError = document.getElementById('defaultsError');
  const defaultsSuccess = document.getElementById('defaultsSuccess');

  function showDefaultsError(msg) {
    defaultsError.textContent = msg;
    defaultsError.style.display = 'block';
    defaultsSuccess.style.display = 'none';
    window.scrollTo(0, 0);
  }

  function showDefaultsSuccess(msg) {
    defaultsSuccess.textContent = msg;
    defaultsSuccess.style.display = 'block';
    defaultsError.style.display = 'none';
    window.scrollTo(0, 0);
    setTimeout(function () {
      defaultsSuccess.style.transition = 'opacity 0.5s ease';
      defaultsSuccess.style.opacity = '0';
      setTimeout(function () {
        defaultsSuccess.style.display = 'none';
        defaultsSuccess.style.opacity = '1';
      }, 500);
    }, 8000);
  }

  if (defaultsForm) {
    defaultsForm.addEventListener('submit', async function (e) {
      e.preventDefault();

      const formData = new FormData(defaultsForm);
      const submitBtn = defaultsForm.querySelector('button[type="submit"]');

      if (submitBtn) {
        submitBtn.disabled = true;
        submitBtn.textContent = 'Saving...';
      }

      try {
        const response = await fetch('/admin/settings.php', {
          method: 'POST',
          headers: {
            'X-Requested-With': 'XMLHttpRequest'
          },
          body: formData,
          credentials: 'same-origin'
        });

        const data = await response.json();

        if (data.ok) {
          showDefaultsSuccess(data.message || 'Settings saved successfully!');
        } else {
          showDefaultsError(data.error || 'Failed to save settings');
        }
      } catch (error) {
        showDefaultsError('Network error. Please try again.');
      } finally {
        if (submitBtn) {
          submitBtn.disabled = false;
          submitBtn.textContent = 'Save Settings';
        }
      }
    });
  }

  // =========================
  // Facebook Protection
  // =========================

  const facebookFrm = document.getElementById('facebookFrm');
  const fbError = document.getElementById('fbError');
  const fbSuccess = document.getElementById('fbSuccess');

  function showFbError(msg) {
    fbError.textContent = msg;
    fbError.style.display = 'block';
    fbSuccess.style.display = 'none';
  }

  function showFbSuccess(msg) {
    fbSuccess.textContent = msg;
    fbSuccess.style.display = 'block';
    fbError.style.display = 'none';
    setTimeout(function () {
      fbSuccess.style.transition = 'opacity 0.5s ease';
      fbSuccess.style.opacity = '0';
      setTimeout(function () {
        fbSuccess.style.display = 'none';
        fbSuccess.style.opacity = '1';
      }, 500);
    }, 8000);
  }

  async function loadFacebookSettings() {
    try {
      const r = await apiGet('/admin/api.php?action=get_facebook_protection_config');
      if (r && r.ok) {
        const cfg = r.config || {};

        document.getElementById('fb_enabled').checked = cfg.enabled !== false;

        document.getElementById('fb_rate_enabled').checked = cfg.rate_limit_enabled !== false;
        document.getElementById('fb_rate_window').value = cfg.rate_limit_window || 60;
        document.getElementById('fb_rate_max').value = cfg.rate_limit_max_requests || 30;
        document.getElementById('fb_rate_suspicious').value = cfg.rate_limit_suspicious_threshold || 10;
        document.getElementById('fb_rate_cooldown').value = cfg.rate_limit_violation_cooldown || 600;

        document.getElementById('fb_dest_enabled').checked = cfg.destination_check_enabled !== false;
        document.getElementById('fb_dest_block_tlds').checked = cfg.destination_check_block_suspicious_tlds !== false;
        document.getElementById('fb_dest_tlds').value = (cfg.destination_check_suspicious_tlds || []).join(',');
        document.getElementById('fb_dest_scan_path').checked = cfg.destination_check_scan_path_patterns !== false;
        document.getElementById('fb_dest_keywords').value = (cfg.destination_check_malicious_keywords || []).join(',');

        document.getElementById('fb_ip_enabled').checked = cfg.ip_validation_enabled !== false;
        document.getElementById('fb_ip_strict').checked = cfg.ip_validation_strict_mode === true;
        document.getElementById('fb_ip_log_suspicious').checked = cfg.ip_validation_log_suspicious !== false;

        document.getElementById('fb_health_enabled').checked = cfg.health_check_enabled !== false;
        document.getElementById('fb_health_cache_ttl').value = cfg.health_check_cache_ttl || 300;
        document.getElementById('fb_health_ban_duration').value = cfg.health_check_ban_duration || 3600;
        document.getElementById('fb_health_cooldown_duration').value = cfg.health_check_cooldown_duration || 600;

        document.getElementById('fb_headers_vary').checked = cfg.response_headers_vary_by_user_agent !== false;
        document.getElementById('fb_headers_allow_bot_cache').checked = cfg.response_headers_allow_bot_caching !== false;
        document.getElementById('fb_headers_bot_cache_ttl').value = cfg.response_headers_bot_cache_ttl || 300;

        document.getElementById('fb_log_enabled').checked = cfg.logging_enabled !== false;
        document.getElementById('fb_log_violations').checked = cfg.logging_log_violations !== false;
        document.getElementById('fb_log_destination_blocks').checked = cfg.logging_log_destination_blocks !== false;
      }
    } catch (e) {
      showFbError('Failed to load settings');
    }
  }

  if (facebookFrm) {
    facebookFrm.addEventListener('submit', async function (ev) {
      ev.preventDefault();

      const payload = {
        enabled: document.getElementById('fb_enabled').checked,
        rate_limit_enabled: document.getElementById('fb_rate_enabled').checked,
        rate_limit_window: parseInt(document.getElementById('fb_rate_window').value, 10) || 60,
        rate_limit_max_requests: parseInt(document.getElementById('fb_rate_max').value, 10) || 30,
        rate_limit_suspicious_threshold: parseInt(document.getElementById('fb_rate_suspicious').value, 10) || 10,
        rate_limit_violation_cooldown: parseInt(document.getElementById('fb_rate_cooldown').value, 10) || 600,

        destination_check_enabled: document.getElementById('fb_dest_enabled').checked,
        destination_check_block_suspicious_tlds: document.getElementById('fb_dest_block_tlds').checked,
        destination_check_suspicious_tlds: (document.getElementById('fb_dest_tlds').value || '')
          .split(',').map(function (s) { return s.trim(); }).filter(function (s) { return s; }),
        destination_check_scan_path_patterns: document.getElementById('fb_dest_scan_path').checked,
        destination_check_malicious_keywords: (document.getElementById('fb_dest_keywords').value || '')
          .split(',').map(function (s) { return s.trim(); }).filter(function (s) { return s; }),

        ip_validation_enabled: document.getElementById('fb_ip_enabled').checked,
        ip_validation_strict_mode: document.getElementById('fb_ip_strict').checked,
        ip_validation_log_suspicious: document.getElementById('fb_ip_log_suspicious').checked,

        health_check_enabled: document.getElementById('fb_health_enabled').checked,
        health_check_cache_ttl: parseInt(document.getElementById('fb_health_cache_ttl').value, 10) || 300,
        health_check_ban_duration: parseInt(document.getElementById('fb_health_ban_duration').value, 10) || 3600,
        health_check_cooldown_duration: parseInt(document.getElementById('fb_health_cooldown_duration').value, 10) || 600,

        response_headers_vary_by_user_agent: document.getElementById('fb_headers_vary').checked,
        response_headers_allow_bot_caching: document.getElementById('fb_headers_allow_bot_cache').checked,
        response_headers_bot_cache_ttl: parseInt(document.getElementById('fb_headers_bot_cache_ttl').value, 10) || 300,

        logging_enabled: document.getElementById('fb_log_enabled').checked,
        logging_log_violations: document.getElementById('fb_log_violations').checked,
        logging_log_destination_blocks: document.getElementById('fb_log_destination_blocks').checked
      };

      const r = await apiPost('update_facebook_protection_config', payload);
      if (r && r.ok) {
        showFbSuccess('Facebook Protection settings saved!');
        window.scrollTo(0, 0);
      } else {
        showFbError((r && r.error) ? r.error : 'Failed to save');
      }
    });
  }

  const btnFbReload = document.getElementById('btnFbReload');
  if (btnFbReload) {
    btnFbReload.addEventListener('click', function () {
      loadFacebookSettings();
    });
  }

  const facebookTab = document.querySelector('.tab[data-tab="facebook"]');
  let facebookLoaded = false;
  if (facebookTab) {
    facebookTab.addEventListener('click', function () {
      if (!facebookLoaded) {
        loadFacebookSettings();
        facebookLoaded = true;
      }
    });
  }

  // =========================
  // Shim Config
  // =========================

  const shimFrm = document.getElementById('shimFrm');
  const shimError = document.getElementById('shimError');
  const shimSuccess = document.getElementById('shimSuccess');

  function showShimError(msg) {
    shimError.textContent = msg;
    shimError.style.display = 'block';
    shimSuccess.style.display = 'none';
  }

  function showShimSuccess(msg) {
    shimSuccess.textContent = msg;
    shimSuccess.style.display = 'block';
    shimError.style.display = 'none';
    setTimeout(function () {
      shimSuccess.style.transition = 'opacity 0.5s ease';
      shimSuccess.style.opacity = '0';
      setTimeout(function () {
        shimSuccess.style.display = 'none';
        shimSuccess.style.opacity = '1';
      }, 500);
    }, 8000);
  }

  async function loadShimSettings() {
    try {
      const r = await apiGet('/admin/api.php?action=get_shim_config');
      if (r && r.ok) {
        const cfg = r.config || {};
        document.getElementById('shim_secret').value = '';
        document.getElementById('shim_secret_hint').textContent = cfg.SHIM_HMAC_SECRET_set ? 'Secret configured (not shown)' : 'Secret not set';
        document.getElementById('shim_allowed_hosts').value = cfg.SHIM_ALLOWED_HOSTS || '';
        document.getElementById('shim_allowed_suffixes').value = cfg.SHIM_ALLOWED_SUFFIXES || '';
        document.getElementById('shim_version').value = cfg.SHIM_VERSION || 'v2';
      }
    } catch (e) {
      showShimError('Failed to load settings');
    }
  }

  if (shimFrm) {
    shimFrm.addEventListener('submit', async function (ev) {
      ev.preventDefault();

      const secret = (document.getElementById('shim_secret').value || '').trim();
      if (secret !== '' && secret.length < 32) {
        showShimError('SHIM_HMAC_SECRET must be at least 32 characters');
        return;
      }

      const payload = {
        SHIM_HMAC_SECRET: secret, // empty => keep existing
        SHIM_ALLOWED_HOSTS: (document.getElementById('shim_allowed_hosts').value || '').trim(),
        SHIM_ALLOWED_SUFFIXES: (document.getElementById('shim_allowed_suffixes').value || '').trim(),
        SHIM_VERSION: document.getElementById('shim_version').value
      };

      const r = await apiPost('update_env_vars', payload);
      if (r && r.ok) {
        showShimSuccess('Shim settings saved!');
        window.scrollTo(0, 0);
        await loadShimSettings();
      } else {
        showShimError((r && r.error) ? r.error : 'Failed to save');
      }
    });
  }

  const btnShimReload = document.getElementById('btnShimReload');
  if (btnShimReload) {
    btnShimReload.addEventListener('click', function () {
      loadShimSettings();
    });
  }

  const shimTab = document.querySelector('.tab[data-tab="shim"]');
  let shimLoaded = false;
  if (shimTab) {
    shimTab.addEventListener('click', function () {
      if (!shimLoaded) {
        loadShimSettings();
        shimLoaded = true;
      }
    });
  }

  // =========================
  // Monitoring
  // =========================

  async function loadMonitoringMetrics() {
    const fbMetrics = document.getElementById('fbMetrics');
    const shimMetrics = document.getElementById('shimMetrics');

    try {
      const r = await apiGet('/admin/api.php?action=get_protection_metrics');
      if (r && r.ok) {
        const fb = r.facebook || {};
        fbMetrics.innerHTML =
          '<div>Rate Limit Exceeded: ' + (fb.rate_limit_exceeded || 0) + '</div>' +
          '<div>Destination Blocks: ' + (fb.destination_blocks || 0) + '</div>' +
          '<div>Suspicious Requests: ' + (fb.suspicious_requests || 0) + '</div>' +
          '<div>Violations: ' + (fb.violations || 0) + '</div>';

        const shim = r.shim || {};
        shimMetrics.innerHTML =
          '<div>Routing Decisions: ' + (shim.routing_decision || 0) + '</div>' +
          '<div>Rate Limit Exceeded: ' + (shim.rate_limit_exceeded || 0) + '</div>' +
          '<div>Fallback Shown: ' + (shim.fallback_shown || 0) + '</div>' +
          '<div>Signature Mismatches: ' + (shim.signature_mismatch || 0) + '</div>';
      } else {
        fbMetrics.textContent = 'Failed to load metrics';
        shimMetrics.textContent = 'Failed to load metrics';
      }
    } catch (e) {
      fbMetrics.textContent = 'Error loading metrics';
      shimMetrics.textContent = 'Error loading metrics';
    }
  }

  async function checkDomainHealth() {
    const domain = (document.getElementById('monitor_domain').value || '').trim();
    const output = document.getElementById('domainHealth');

    if (!domain) {
      output.textContent = 'Please enter a domain';
      return;
    }

    output.textContent = 'Checking ' + domain + '...';

    try {
      const r = await apiGet('/admin/api.php?action=check_domain_health&domain=' + encodeURIComponent(domain));
      if (r && r.ok) {
        const h = r.health || {};
        output.innerHTML =
          '<div>Domain: ' + domain + '</div>' +
          '<div>Requests/min: ' + (h.requests_per_minute || 0) + '</div>' +
          '<div>Violations: ' + (h.violations ? 'YES' : 'NO') + '</div>' +
          '<div>Banned: ' + (h.banned ? 'YES' : 'NO') + '</div>' +
          '<div>Cooldown: ' + (h.cooldown ? 'YES' : 'NO') + '</div>';
      } else {
        output.textContent = 'Error: ' + ((r && r.error) || 'Unknown error');
      }
    } catch (e) {
      output.textContent = 'Error checking domain';
    }
  }

  const btnRefreshMetrics = document.getElementById('btnRefreshMetrics');
  if (btnRefreshMetrics) {
    btnRefreshMetrics.addEventListener('click', function () {
      loadMonitoringMetrics();
    });
  }

  const btnCheckDomain = document.getElementById('btnCheckDomain');
  if (btnCheckDomain) {
    btnCheckDomain.addEventListener('click', function () {
      checkDomainHealth();
    });
  }

  const monitoringTab = document.querySelector('.tab[data-tab="monitoring"]');
  let monitoringLoaded = false;
  if (monitoringTab) {
    monitoringTab.addEventListener('click', function () {
      if (!monitoringLoaded) {
        loadMonitoringMetrics();
        monitoringLoaded = true;
      }
    });
  }
})();
</script>
</body>
</html>
