<?php

declare(strict_types=1);

/**
 * Shortlinks Management Page
 * Dedicated page untuk manage shortlinks
 */

use App\Admin\Auth;
use App\Admin\Security;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';

Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();
$csrf = Auth::ensureCsrfToken();
$isAdmin = Auth::isAdmin();

$publicBase = rtrim(Env::getString('PUBLIC_BASE_URL', ''), '/');

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <meta name="public-base" content="<?= e($publicBase) ?>">
  <title>Shortlinks | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">Dashboard</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php" aria-current="page">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/users.php">Users</a>
        <?php endif; ?>
        <a href="/admin/domains.php">Domains</a>
        <?php if ($isAdmin) : ?>
        <a href="/admin/global-domains.php">Global Domains</a>
        <?php endif; ?>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">Shortlinks</h1>
            <div class="page-sub">Manage all shortlinks</div>
          </div>
          <div class="topbar-actions">
            <button type="button" class="btn btn-primary" id="btnCreateNew">+ Create Shortlink</button>
          </div>
        </div>

        <!-- Filters & Search -->
        <div class="card" style="margin-bottom: 20px;">
          <div style="display: flex; gap: 12px; align-items: center; flex-wrap: wrap;">
            <div style="flex: 1; min-width: 200px;">
              <input type="text" id="searchCode" placeholder="Search by code..." style="margin: 0;">
            </div>
            <div>
              <select id="filterActive">
                <option value="">All Status</option>
                <option value="1" selected>Active Only</option>
                <option value="0">Inactive Only</option>
              </select>
            </div>
            <div>
              <select id="filterShim">
                <option value="">All Shim</option>
                <option value="1">Shim Enabled</option>
                <option value="0">Shim Disabled</option>
              </select>
            </div>
            <button type="button" class="btn btn-secondary" id="btnSearch">Search</button>
            <button type="button" class="btn" id="btnReset">Reset</button>
          </div>
        </div>

        <!-- Shortlinks Table -->
        <div class="card">
          <div id="shortlinksTable">
            <div style="text-align: center; padding: 40px; color: #666;">
              Loading shortlinks...
            </div>
          </div>

          <!-- Pagination -->
          <div id="pagination" style="display: none; padding: 16px; border-top: 1px solid var(--border); text-align: center;">
            <button type="button" class="btn btn-sm" id="btnPrevPage" disabled>Previous</button>
            <span id="pageInfo" style="margin: 0 16px; color: #666;"></span>
            <button type="button" class="btn btn-sm" id="btnNextPage" disabled>Next</button>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- Create/Edit Modal -->
  <div id="modalShortlink" class="modal" style="display: none;">
    <div class="modal-content" style="max-width: 700px;">
      <div class="modal-header">
        <h3 id="modalTitle">Create Shortlink</h3>
        <button type="button" class="modal-close" id="modalClose">&times;</button>
      </div>
      <form id="frmShortlink" autocomplete="off">
        <input type="hidden" id="editMode" value="0">
        <input type="hidden" id="originalCode" value="">

        <div class="form-group">
          <label for="code">Shortlink Code *</label>
          <input type="text" id="code" required placeholder="e.g., promo2024">
          <small>3-32 alphanumeric characters, hyphens, underscores</small>
        </div>

        <div class="form-group">
          <label for="target_url">Target URL *</label>
          <input type="url" id="target_url" required placeholder="https://example.com/page">
        </div>

        <div class="form-group">
          <label for="og_title">OG Title *</label>
          <input type="text" id="og_title" required placeholder="Page Title">
        </div>

        <div class="form-group">
          <label for="og_description">OG Description *</label>
          <textarea id="og_description" required rows="3" placeholder="Page description"></textarea>
        </div>

        <div class="form-group">
          <label for="og_image_url">OG Image URL</label>
          <input type="url" id="og_image_url" placeholder="https://example.com/image.jpg">
        </div>

        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px;">
          <div class="form-group">
            <label for="og_type">OG Type</label>
            <select id="og_type">
              <option value="website">Website</option>
              <option value="article">Article</option>
              <option value="product">Product</option>
            </select>
          </div>

          <div class="form-group">
            <label for="og_locale">OG Locale</label>
            <input type="text" id="og_locale" value="id_ID" placeholder="id_ID">
          </div>
        </div>

        <div class="form-group">
          <label for="og_site_name">OG Site Name</label>
          <input type="text" id="og_site_name" placeholder="Site Name">
        </div>

        <div class="form-group">
          <label>
            <input type="checkbox" id="use_shim">
            Enable Shim Mode (bot protection)
          </label>
        </div>

        <div class="form-group">
          <label>
            <input type="checkbox" id="is_active" checked>
            Active
          </label>
        </div>

        <div class="modal-footer">
          <button type="button" class="btn" id="btnCancel">Cancel</button>
          <button type="submit" class="btn btn-primary">Save</button>
        </div>
      </form>
    </div>
  </div>

  <script nonce="<?= e($nonce) ?>">
    const csrfToken = document.querySelector('meta[name="csrf-token"]').content;
    const publicBase = document.querySelector('meta[name="public-base"]').content;

    let currentPage = 1;
    let pageSize = 20;
    let totalItems = 0;
    let currentFilters = {};

    // API helpers
    async function apiGet(url) {
      const r = await fetch(url, { headers: { 'X-CSRF-Token': csrfToken } });
      return r.json();
    }

    async function apiPost(action, data) {
      const r = await fetch('/admin/api.php?action=' + action, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'X-CSRF-Token': csrfToken },
        body: JSON.stringify(data)
      });
      return r.json();
    }

    // Load shortlinks
    async function loadShortlinks() {
      const params = new URLSearchParams({
        page: currentPage,
        page_size: pageSize,
        ...currentFilters
      });

      const r = await apiGet(`/admin/api.php?action=get_shortlinks&${params}`);

      if (r && r.ok) {
        totalItems = r.total || 0;
        renderTable(r.data || []);
        updatePagination();
      } else {
        document.getElementById('shortlinksTable').innerHTML = '<div style="padding: 40px; text-align: center; color: #e74c3c;">Failed to load shortlinks</div>';
      }
    }

    // Render table
    function renderTable(items) {
      if (items.length === 0) {
        document.getElementById('shortlinksTable').innerHTML = '<div style="padding: 40px; text-align: center; color: #666;">No shortlinks found</div>';
        return;
      }

      let html = '<table><thead><tr><th>Code</th><th>Title</th><th>Status</th><th>Shim</th><th>Created</th><th>Actions</th></tr></thead><tbody>';

      items.forEach(item => {
        const url = publicBase + '/' + item.code;
        const status = item.is_active ? '<span class="badge badge-success">Active</span>' : '<span class="badge badge-secondary">Inactive</span>';
        const shim = item.use_shim ? '<span class="badge badge-info">Enabled</span>' : '';

        html += `<tr>
          <td><a href="${url}" target="_blank" style="font-family: monospace;">${item.code}</a></td>
          <td>${item.og_title}</td>
          <td>${status}</td>
          <td>${shim}</td>
          <td>${item.created_at}</td>
          <td>
            <button type="button" class="btn btn-sm" onclick="editShortlink('${item.code}')">Edit</button>
            <button type="button" class="btn btn-sm btn-danger" onclick="deleteShortlink('${item.code}')">Delete</button>
          </td>
        </tr>`;
      });

      html += '</tbody></table>';
      document.getElementById('shortlinksTable').innerHTML = html;
    }

    // Update pagination
    function updatePagination() {
      const totalPages = Math.ceil(totalItems / pageSize);
      document.getElementById('pageInfo').textContent = `Page ${currentPage} of ${totalPages} (${totalItems} total)`;
      document.getElementById('btnPrevPage').disabled = currentPage === 1;
      document.getElementById('btnNextPage').disabled = currentPage >= totalPages;
      document.getElementById('pagination').style.display = totalPages > 1 ? 'block' : 'none';
    }

    // Edit shortlink
    window.editShortlink = async function(code) {
      const r = await apiGet(`/admin/api.php?action=get_shortlink&code=${code}`);

      if (r && r.ok && r.data) {
        const d = r.data;
        document.getElementById('modalTitle').textContent = 'Edit Shortlink';
        document.getElementById('editMode').value = '1';
        document.getElementById('originalCode').value = d.code;
        document.getElementById('code').value = d.code;
        document.getElementById('target_url').value = d.target_url;
        document.getElementById('og_title').value = d.og_title;
        document.getElementById('og_description').value = d.og_description;
        document.getElementById('og_image_url').value = d.og_image_url || '';
        document.getElementById('og_type').value = d.og_type || 'website';
        document.getElementById('og_locale').value = d.og_locale || 'id_ID';
        document.getElementById('og_site_name').value = d.og_site_name || '';
        document.getElementById('use_shim').checked = d.use_shim === 1;
        document.getElementById('is_active').checked = d.is_active === 1;
        document.getElementById('modalShortlink').style.display = 'flex';
      }
    };

    // Delete shortlink
    window.deleteShortlink = async function(code) {
      if (!confirm(`Delete shortlink "${code}"?`)) return;

      const r = await apiPost('delete_shortlink', { code });
      if (r && r.ok) {
        alert('Shortlink deleted');
        loadShortlinks();
      } else {
        alert('Failed: ' + (r.error || 'Unknown error'));
      }
    };

    // Event listeners
    document.getElementById('btnCreateNew').addEventListener('click', () => {
      document.getElementById('frmShortlink').reset();
      document.getElementById('modalTitle').textContent = 'Create Shortlink';
      document.getElementById('editMode').value = '0';
      document.getElementById('is_active').checked = true;
      document.getElementById('og_locale').value = 'id_ID';
      document.getElementById('modalShortlink').style.display = 'flex';
    });

    document.getElementById('modalClose').addEventListener('click', () => {
      document.getElementById('modalShortlink').style.display = 'none';
    });

    document.getElementById('btnCancel').addEventListener('click', () => {
      document.getElementById('modalShortlink').style.display = 'none';
    });

    document.getElementById('frmShortlink').addEventListener('submit', async (e) => {
      e.preventDefault();

      const editMode = document.getElementById('editMode').value === '1';
      const payload = {
        code: document.getElementById('code').value.trim(),
        target_url: document.getElementById('target_url').value.trim(),
        og_title: document.getElementById('og_title').value.trim(),
        og_description: document.getElementById('og_description').value.trim(),
        og_image_url: document.getElementById('og_image_url').value.trim(),
        og_type: document.getElementById('og_type').value,
        og_locale: document.getElementById('og_locale').value.trim(),
        og_site_name: document.getElementById('og_site_name').value.trim(),
        use_shim: document.getElementById('use_shim').checked ? 1 : 0,
        is_active: document.getElementById('is_active').checked ? 1 : 0
      };

      if (editMode) {
        payload.original_code = document.getElementById('originalCode').value;
      }

      const action = editMode ? 'update_shortlink' : 'create_shortlink';
      const r = await apiPost(action, payload);

      if (r && r.ok) {
        alert(editMode ? 'Shortlink updated' : 'Shortlink created');
        document.getElementById('modalShortlink').style.display = 'none';
        loadShortlinks();
      } else {
        alert('Failed: ' + (r.error || 'Unknown error'));
      }
    });

    document.getElementById('btnSearch').addEventListener('click', () => {
      currentFilters = {
        search: document.getElementById('searchCode').value.trim(),
        active: document.getElementById('filterActive').value,
        use_shim: document.getElementById('filterShim').value
      };
      currentPage = 1;
      loadShortlinks();
    });

    document.getElementById('btnReset').addEventListener('click', () => {
      document.getElementById('searchCode').value = '';
      document.getElementById('filterActive').value = '1';
      document.getElementById('filterShim').value = '';
      currentFilters = {};
      currentPage = 1;
      loadShortlinks();
    });

    document.getElementById('btnPrevPage').addEventListener('click', () => {
      if (currentPage > 1) {
        currentPage--;
        loadShortlinks();
      }
    });

    document.getElementById('btnNextPage').addEventListener('click', () => {
      const totalPages = Math.ceil(totalItems / pageSize);
      if (currentPage < totalPages) {
        currentPage++;
        loadShortlinks();
      }
    });

    // Initial load
    loadShortlinks();
  </script>

  <style nonce="<?= e($nonce) ?>">
    .modal {
      display: none;
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0, 0, 0, 0.5);
      align-items: center;
      justify-content: center;
      z-index: 1000;
    }

    .modal-content {
      background: white;
      border-radius: 8px;
      width: 90%;
      max-height: 90vh;
      overflow-y: auto;
    }

    .modal-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 20px;
      border-bottom: 1px solid var(--border);
    }

    .modal-header h3 {
      margin: 0;
    }

    .modal-close {
      background: none;
      border: none;
      font-size: 28px;
      cursor: pointer;
      color: #666;
    }

    .modal-close:hover {
      color: #333;
    }

    .modal form {
      padding: 20px;
    }

    .form-group {
      margin-bottom: 16px;
    }

    .form-group label {
      display: block;
      margin-bottom: 4px;
      font-weight: 500;
    }

    .form-group input[type="text"],
    .form-group input[type="url"],
    .form-group textarea,
    .form-group select {
      width: 100%;
      padding: 8px 12px;
      border: 1px solid var(--border);
      border-radius: 4px;
      font-family: inherit;
    }

    .form-group small {
      display: block;
      margin-top: 4px;
      color: #666;
      font-size: 0.85em;
    }

    .modal-footer {
      display: flex;
      justify-content: flex-end;
      gap: 8px;
      padding-top: 16px;
      border-top: 1px solid var(--border);
      margin-top: 16px;
    }

    .badge {
      display: inline-block;
      padding: 4px 8px;
      border-radius: 4px;
      font-size: 0.85em;
      font-weight: 500;
    }

    .badge-success { background: #d4edda; color: #155724; }
    .badge-secondary { background: #e2e3e5; color: #383d41; }
    .badge-info { background: #d1ecf1; color: #0c5460; }

    .btn-sm {
      padding: 4px 8px;
      font-size: 0.85em;
    }

    .btn-danger {
      background: #e74c3c;
      color: white;
    }

    .btn-danger:hover {
      background: #c0392b;
    }

    .topbar-actions {
      display: flex;
      gap: 8px;
    }
  </style>
</body>
</html>
